//////////////////////////////////////////////////////////////////////////////
// uDBS - UScript Database System
//		Feralidragon - 09-07-2013
//
//		uWebDrvX 0.1
//////////////////////////////////////////////////////////////////////////////

class uDBS extends Object config(uDBS)
abstract;

struct uTable
{
	var() name TableClassName;
	var() name TableName;
	var class<uDBSTable> TableClass;
	var string TableQName;
};

var() uTable TablesList[64];
var() name DatabaseName;
var byte dbSize;

enum EQOp
{
	QOP_NOOP,
	QOP_SELECT,
	QOP_COUNT,
	QOP_SUM,
	QOP_MAX,
	QOP_MIN,
	QOP_AVG,
	QOP_UPDATE,
	QOP_INSERT,
	QOP_DELETE
};
var const string EQOpNames[10];

enum EQCond
{
	QC_NoCond,
	QC_Equal,
	QC_NotEqual,
	QC_EqualOrGreaterThan,
	QC_GreaterThan,
	QC_EqualOrLesserThan,
	QC_LesserThan,
	QC_LikeRight,
	QC_LikeLeft,
	QC_LikeBoth,
	QC_ILikeRight,
	QC_ILikeLeft,
	QC_ILikeBoth
};
var const string EQCondNames[13];

struct uQueryCond
{
	var EQCond Cond;
	var byte colIndex;
	var byte paramIndex;
};

struct uQueryORCond
{
	var uQueryCond Cond1;
	var uQueryCond Cond2;
};

struct uQuery
{
	var EQOp Operation;
	var byte returnCols[32];
	var byte returnColsSize;
	var class<uDBSTable> Table;
	var uQueryCond WhereAndConds[8];
	var byte WhereAndCondsSize;
	var uQueryORCond WhereOrConds[8];
	var byte WhereOrCondsSize;
};


////////////////////////////////////////////////////////////////////////////////////////////////////
// LOAD DB
////////////////////////////////////////////////////////////////////////////////////////////////////
final static function bool loadDB()
{
local byte i, j;
local string Pkg;
	
	default.dbSize = 0;
	Pkg = Class'uUtils'.static.getPackageName(String(default.Class));
	for (i = 0; i < ArrayCount(default.TablesList); i++)
	{
		if (default.TablesList[i].TableName != '')
		{
			for (j = i + 1; j < ArrayCount(default.TablesList); j++)
			{
				if (default.TablesList[j].TableName != '' && default.TablesList[j].TableName == default.TablesList[i].TableName)
				{
					log("ERROR: Database could not be loaded: duplicated table name for '"$default.TablesList[i].TableName$"'.", default.DatabaseName);
					return False;
				}
			}
		}
	}
	
	for (i = 0; i < ArrayCount(default.TablesList); i++)
	{
		if (default.TablesList[i].TableClassName != '')
		{
			if (default.TablesList[i].TableName == '')
			{
				log("WARNING: No table name found for '"$default.TablesList[i].TableClassName$"'.", default.DatabaseName);
				default.TablesList[i].TableName = default.TablesList[i].TableClassName;
			}
			default.TablesList[i].TableClass = class<uDBSTable>(DynamicLoadObject(Pkg$"."$default.TablesList[i].TableClassName, class'Class'));
			default.TablesList[i].TableQName = Caps(String(default.TablesList[i].TableName));
		}

		if (default.TablesList[i].TableClass == None)
		{
			if (default.TablesList[i].TableClassName != '')
			{
				log("ERROR: Database could not be loaded: table class name for '"$default.TablesList[i].TableClassName$"' not found.", default.DatabaseName);
				return False;
			}
			default.dbSize = i;
			break;
		}
		
		if (i == (ArrayCount(default.TablesList) - 1))
			default.dbSize = i + 1;
	}
	
	for (i = 0; i < default.dbSize; i++)
	{
		if (!default.TablesList[i].TableClass.static.loadTable(default.Class))
		{
			log("ERROR: Database could not be loaded: failed to load table '"$default.TablesList[i].TableName$"'.", default.DatabaseName);
			return False;
		}
	}
	
	return True;
}

final static function bool isLoadedDB()
{
	return (default.dbSize > 0);
}

final static function string query(string qString, optional string qP00, optional string qP01, optional string qP02, optional string qP03, 
optional string qP04, optional string qP05, optional string qP06, optional string qP07)
{
local string qParams[32];

	qParams[0]=qP00; qParams[1]=qP01; qParams[2]=qP02; qParams[3]=qP03; 
	qParams[4]=qP04; qParams[5]=qP05; qParams[6]=qP06; qParams[7]=qP07;
	return query2(qString, qParams);
}

final static function string query2(string qString, optional string qParams[32])
{
local uQuery uQ;

	if (!isLoadedDB())
	{
		log("ERROR: Query failed: database not loaded.", default.DatabaseName);
		return "";
	}
	
	uQ = getCompiledQuery(qString);
	uQueryDump(uQ);
}

final static function uQuery getCompiledQuery(string qString)
{
local uQuery uQ, nullQ;
local int i;
local bool getAll, continueLoop;
local string qStr;
local byte curParamIndex;

	qString = Caps(qString);
	Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
	
	//Check query operation type
	if (InStr(qString, "SELECT") == 0)
	{
		uQ.Operation = QOP_SELECT;
		qString = Mid(qString, Len("SELECT"));
		if (Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
		{
			if (InStr(qString, "COUNT(") == 0)
			{
				uQ.Operation = QOP_COUNT;
				qString = Mid(qString, Len("COUNT("));
			}
			else if (InStr(qString, "SUM(") == 0)
			{
				uQ.Operation = QOP_SUM;
				qString = Mid(qString, Len("SUM("));
			}
			else if (InStr(qString, "MAX(") == 0)
			{
				uQ.Operation = QOP_MAX;
				qString = Mid(qString, Len("MAX("));
			}
			else if (InStr(qString, "MIN(") == 0)
			{
				uQ.Operation = QOP_MIN;
				qString = Mid(qString, Len("MIN("));
			}
			else if (InStr(qString, "AVG(") == 0)
			{
				uQ.Operation = QOP_AVG;
				qString = Mid(qString, Len("AVG("));
			}
		}
		else
			uQ.Operation = QOP_NOOP;
	}
	else if (InStr(qString, "UPDATE") == 0)
	{
		uQ.Operation = QOP_UPDATE;
		qString = Mid(qString, Len("UPDATE"));
	}
	else if (InStr(qString, "INSERT") == 0)
	{
		uQ.Operation = QOP_INSERT;
		qString = Mid(qString, Len("INSERT"));
	}
	else if (InStr(qString, "DELETE") == 0)
	{
		uQ.Operation = QOP_DELETE;
		qString = Mid(qString, Len("DELETE"));
	}
	
	if (uQ.Operation == QOP_NOOP)
	{
		log("ERROR: Query syntax error: queries must start with SELECT, UPDATE, INSERT or DELETE.", default.DatabaseName);
		return nullQ;
	}
	
	
	//Check syntax and fields of query
	if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString) && uQ.Operation >= QOP_UPDATE)
	{
		log("ERROR: Query syntax error on UPDATE.", default.DatabaseName);
		return nullQ;
	}
	
	
	//Handle UPDATE
	if (uQ.Operation == QOP_UPDATE)
	{
		_processUQTableClass(uQ, qString);
		if (uQ.Table == None)
			return nullQ;
		
		if (InStr(qString, "SET") != 0)
		{
			log("ERROR: Query syntax error: SET not found.", default.DatabaseName);
			return nullQ;
		}
		
		qString = Mid(qString, Len("SET"));
		if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
		{
			log("ERROR: Query syntax error on SET.", default.DatabaseName);
			return nullQ;
		}
		
		_processUQReturnCols(uQ, qString, True);
		if (uQ.returnColsSize == 0)
			return nullQ;
		curParamIndex = uQ.returnColsSize;
	}
	
	//Handle INSERT
	else if (uQ.Operation == QOP_INSERT)
	{
		if (InStr(qString, "INTO") != 0)
		{
			log("ERROR: Query syntax error: INTO not found.", default.DatabaseName);
			return nullQ;
		}
		
		qString = Mid(qString, Len("INTO"));
		if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
		{
			log("ERROR: Query syntax error: INTO", default.DatabaseName);
			return nullQ;
		}
		
		_processUQTableClass(uQ, qString);
		if (uQ.Table == None)
			return nullQ;
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		if (InStr(qString, "(") != 0)
		{
			log("ERROR: Query syntax error: no entry ( found.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, 1);
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		_processUQReturnCols(uQ, qString);
		if (uQ.returnColsSize == 0)
			return nullQ;
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		if (InStr(qString, ")") != 0)
		{
			log("ERROR: Query syntax error: no enclosing ) found.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, 1);
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		if (InStr(qString, "VALUES") != 0)
		{
			log("ERROR: Query syntax error: no VALUES found.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, Len("VALUES"));
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		if (InStr(qString, "(") != 0)
		{
			log("ERROR: Query syntax error: no entry ( found.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, 1);
		
		for (i = 0; i < uQ.returnColsSize; i++)
		{
			Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			if (InStr(qString, "?") != 0)
			{
				log("ERROR: Query syntax error: INSERT parameter ? expected.", default.DatabaseName);
				return nullQ;
			}
			qString = Mid(qString, 1);
			
			if (i < (uQ.returnColsSize - 1))
			{
				Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
				if (InStr(qString, ",") != 0)
				{
					log("ERROR: Query syntax error: INSERT separator , expected.", default.DatabaseName);
					return nullQ;
				}
				qString = Mid(qString, 1);
			}
		}
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		if (InStr(qString, ")") != 0)
		{
			log("ERROR: Query syntax error: no enclosing ) found.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, 1);
		curParamIndex = uQ.returnColsSize;
	}
	
	//Handle DELETE
	else if (uQ.Operation == QOP_DELETE)
	{
		if (InStr(qString, "FROM") != 0)
		{
			log("ERROR: Query syntax error: FROM not found.", default.DatabaseName);
			return nullQ;
		}
		
		qString = Mid(qString, Len("FROM"));
		if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
		{
			log("ERROR: Query syntax error: FROM", default.DatabaseName);
			return nullQ;
		}
		
		_processUQTableClass(uQ, qString);
		if (uQ.Table == None)
			return nullQ;
	}
	
	//Handle SELECT and derivatives
	else if (uQ.Operation >= QOP_SELECT)
	{
		if (InStr(qString, "*") == 0)
		{
			qString = Mid(qString, 1);
			if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
			{
				log("ERROR: Query syntax error: *", default.DatabaseName);
				return nullQ;
			}
			getAll = True;
		}
		else
		{
			qStr = qString;
			_skipColumnsDecls(qString);
		}

		if (InStr(qString, "FROM") != 0)
		{
			log("ERROR: Query syntax error: FROM not found.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, Len("FROM"));
		
		if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
		{
			log("ERROR: Query syntax error: FROM", default.DatabaseName);
			return nullQ;
		}
		
		_processUQTableClass(uQ, qString);
		if (uQ.Table == None)
			return nullQ;
			
		if (getAll)
		{
			uQ.returnColsSize = uQ.Table.static.getColsSize();
			for (i = 0; i < uQ.returnColsSize; i++)
				uQ.returnCols[i] = i;
		}
		else
		{
			qString = qStr;
			_processUQReturnCols(uQ, qString);
			if (uQ.returnColsSize == 0)
				return nullQ;

			if (uQ.Operation >= QOP_COUNT)
			{
				if (uQ.returnColsSize > 1)
				{
					log("ERROR: Query error: aggregate functions cannot return more than 1 column.", default.DatabaseName);
					return nullQ;
				}
				if (InStr(qString, ")") != 0)
				{
					log("ERROR: Query syntax error: no enclosing ) found.", default.DatabaseName);
					return nullQ;
				}
				qString = Mid(qString, 1);
				Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			}
			
			if (InStr(qString, "FROM") != 0)
			{
				log("ERROR: Query syntax error: FROM not found.", default.DatabaseName);
				return nullQ;
			}
			qString = Mid(qString, Len("FROM"));
			
			if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
			{
				log("ERROR: Query syntax error: FROM", default.DatabaseName);
				return nullQ;
			}
				
			_processUQTableClass(uQ, qString);
			if (uQ.Table == None)
				return nullQ;
		}
	}
	
	Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
	if (_isEndOfQuery(qString))
		return uQ;
	if (uQ.Operation == QOP_INSERT)
	{
		log("ERROR: Query syntax error: expected end of query.", default.DatabaseName);
		return nullQ;
	}
	
	
	//Handle WHERE or end of query
	if (InStr(qString, "WHERE") != 0)
	{
		log("ERROR: Query syntax error: WHERE not found.", default.DatabaseName);
		return nullQ;
	}
	qString = Mid(qString, Len("WHERE"));
	
	if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
	{
		log("ERROR: Query syntax error: WHERE", default.DatabaseName);
		return nullQ;
	}
	
	do
	{
		continueLoop = False;
		if (InStr(qString, "(") == 0)
		{
			qString = Mid(qString, 1);
			Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			_processUQORCond(uQ, qString, curParamIndex);
			if (uQ.WhereOrCondsSize == 0)
				return nullQ;
				
			if (InStr(qString, ")") != 0)
			{
				log("ERROR: Query syntax error: no enclosing OR ) found.", default.DatabaseName);
				return nullQ;
			}
			qString = Mid(qString, 1);
		}
		else
		{
			_processUQANDCond(uQ, qString, curParamIndex);
			if (uQ.WhereAndCondsSize == 0)
				return nullQ;
		}
		
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		if (_isEndOfQuery(qString))
			return uQ;
		if (InStr(qString, "AND") != 0)
		{
			log("ERROR: Query syntax error: expected AND operator.", default.DatabaseName);
			return nullQ;
		}
		qString = Mid(qString, Len("AND"));

		if (!Class'uUtils'.static.stripLeadingWhiteSpaces(qString))
		{
			log("ERROR: Query syntax error: AND.", default.DatabaseName);
			return nullQ;
		}
		continueLoop = True;
	}
	until (!continueLoop);
	
	return uQ;
}

final static function _skipColumnsDecls(out string qString)
{
local int cpos;

	do
	{
		Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
		cpos = InStr(qString, ",");
		if (cpos > 0)
			qString = Mid(qString, cpos + 1);
		else
		{
			cpos = Class'uUtils'.static.getClosestWhiteSpacePos(qString);
			if (cpos < 0)
				return;
			qString = Mid(qString, cpos + 1);
			Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			cpos = -1;
		}
	}
	until (cpos < 0);
}

final static function _processUQTableClass(out uQuery uQ, out string qString)
{
local string curTableQName;
local int i;

	for (i = 0; i < default.dbSize; i++)
	{
		if (InStr(qString, default.TablesList[i].TableQName) == 0 && Len(curTableQName) < Len(default.TablesList[i].TableQName))
		{
			curTableQName = default.TablesList[i].TableQName;
			uQ.Table = default.TablesList[i].TableClass;
		}
	}
	qString = Mid(qString, Len(curTableQName));
	
	if (uQ.Table == None || (!_isEndOfQuery(qString) && !Class'uUtils'.static.stripLeadingWhiteSpaces(qString)))
	{
		uQ.Table = None;
		log("ERROR: No valid table name found in query.", default.DatabaseName);
	}
}

final static function bool _isEndOfQuery(string qString)
{
	return (qString == "" || InStr(qString, ";") == 0);
}

final static function _processUQReturnCols(out uQuery uQ, out string qString, optional bool bUpdateOp)
{
local int i, j, sz;
local string str, savedColQName;
local bool continueLoop;
local byte savedColN;

	j = 0;
	do
	{
		if (uQ.returnColsSize >= ArrayCount(uQ.returnCols))
		{
			log("ERROR: Query error: UPDATE columns limit overflow.", default.DatabaseName);
			uQ.returnColsSize = 0;
			return;
		}

		continueLoop = False;
		sz = uQ.Table.static.getColsSize();
		savedColQName = "";
		savedColN = 0;
		for (i = 0; i < sz; i++)
		{
			str = uQ.Table.static.getColsQName(i);
			if (InStr(qString, str) == 0 && Len(savedColQName) < Len(str))
			{
				savedColQName = str;
				savedColN = i;
			}
		}
		
		if (savedColQName != "")
		{
			uQ.returnCols[j] = savedColN;
			uQ.returnColsSize = (++j);
			qString = Mid(qString, Len(savedColQName));
			Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			if (bUpdateOp)
			{
				if (InStr(qString, "=") != 0)
				{
					log("ERROR: Query syntax error: no valid COLUMN= found.", default.DatabaseName);
					uQ.returnColsSize = 0;
					return;
				}
				qString = Mid(qString, 1);
				Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
				if (InStr(qString, "?") != 0)
				{
					log("ERROR: Query syntax error: no valid COLUMN=? found.", default.DatabaseName);
					uQ.returnColsSize = 0;
					return;
				}
				qString = Mid(qString, 1);
				Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			}
			
			if (InStr(qString, ",") == 0)
			{
				continueLoop = True;
				qString = Mid(qString, 1);
				Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
			}
		}
	}
	until (!continueLoop);
	
	if (uQ.returnColsSize == 0)
		log("ERROR: Query error: no columns.", default.DatabaseName);
}

final static function _processUQORCond(out uQuery uQ, out string qString, out byte curParamIndex)
{
local uQueryCond uQC;

	if (uQ.WhereOrCondsSize >= ArrayCount(uQ.WhereOrConds))
	{
		log("ERROR: Query error: WHERE OR limit overflow.", default.DatabaseName);
		uQ.WhereOrCondsSize = 0;
		return;
	}
	
	uQC = _processUQCond(uQ, qString, curParamIndex);
	uQ.WhereOrConds[uQ.WhereOrCondsSize].Cond1 = uQC;
	if (uQ.WhereOrConds[uQ.WhereOrCondsSize].Cond1.Cond == QC_NoCond)
	{
		uQ.WhereOrCondsSize = 0;
		return;
	}
	
	if (InStr(qString, "OR") != 0)
	{
		log("ERROR: Query syntax error: missing OR operator.", default.DatabaseName);
		uQ.WhereOrCondsSize = 0;
		return;
	}
	qString = Mid(qString, 2);
	Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
	
	uQC = _processUQCond(uQ, qString, curParamIndex);
	uQ.WhereOrConds[uQ.WhereOrCondsSize].Cond2 = uQC;
	if (uQ.WhereOrConds[uQ.WhereOrCondsSize].Cond2.Cond == QC_NoCond)
	{
		uQ.WhereOrCondsSize = 0;
		return;
	}
	uQ.WhereOrCondsSize++;
}

final static function _processUQANDCond(out uQuery uQ, out string qString, out byte curParamIndex)
{
local uQueryCond uQC;

	if (uQ.WhereAndCondsSize >= ArrayCount(uQ.WhereAndConds))
	{
		log("ERROR: Query error: WHERE AND limit overflow.", default.DatabaseName);
		uQ.WhereAndCondsSize = 0;
		return;
	}
	
	uQC = _processUQCond(uQ, qString, curParamIndex);
	uQ.WhereAndConds[uQ.WhereAndCondsSize] = uQC;
	if (uQ.WhereAndConds[uQ.WhereAndCondsSize].Cond == QC_NoCond)
	{
		uQ.WhereAndCondsSize = 0;
		return;
	}
	uQ.WhereAndCondsSize++;
}

final static function uQueryCond _processUQCond(out uQuery uQ, out string qString, out byte curParamIndex)
{
local int i, sz;
local string str, savedColQName;
local byte savedColN;
local uQueryCond uQCond;

	sz = uQ.Table.static.getColsSize();
	savedColQName = "";
	savedColN = 0;
	for (i = 0; i < sz; i++)
	{
		str = uQ.Table.static.getColsQName(i);
		if (InStr(qString, str) == 0 && Len(savedColQName) < Len(str))
		{
			savedColQName = str;
			savedColN = i;
		}
	}
	
	if (savedColQName == "")
	{
		log("ERROR: Query error: invalid or missing WHERE condition.", default.DatabaseName);
		uQCond.Cond = QC_NoCond;
		return uQCond;
	}

	uQCond.colIndex = savedColN;
	uQCond.paramIndex = curParamIndex++;
	
	qString = Mid(qString, Len(savedColQName));
	Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
	if (InStr(qString, "=") == 0)
	{
		uQCond.Cond = QC_Equal;
		qString = Mid(qString, 1);
	}
	else if (InStr(qString, "<>") == 0)
	{
		uQCond.Cond = QC_NotEqual;
		qString = Mid(qString, 2);
	}
	else if (InStr(qString, ">=") == 0)
	{
		uQCond.Cond = QC_EqualOrGreaterThan;
		qString = Mid(qString, 2);
	}
	else if (InStr(qString, ">") == 0)
	{
		uQCond.Cond = QC_GreaterThan;
		qString = Mid(qString, 1);
	}
	else if (InStr(qString, "<=") == 0)
	{
		uQCond.Cond = QC_EqualOrLesserThan;
		qString = Mid(qString, 2);
	}
	else if (InStr(qString, "<") == 0)
	{
		uQCond.Cond = QC_LesserThan;
		qString = Mid(qString, 1);
	}
	else if (InStr(qString, "LIKER") == 0)
	{
		uQCond.Cond = QC_LikeRight;
		qString = Mid(qString, 5);
	}
	else if (InStr(qString, "LIKEL") == 0)
	{
		uQCond.Cond = QC_LikeLeft;
		qString = Mid(qString, 5);
	}
	else if (InStr(qString, "LIKE") == 0)
	{
		uQCond.Cond = QC_LikeBoth;
		qString = Mid(qString, 4);
	}
	else if (InStr(qString, "ILIKER") == 0)
	{
		uQCond.Cond = QC_ILikeRight;
		qString = Mid(qString, 6);
	}
	else if (InStr(qString, "ILIKEL") == 0)
	{
		uQCond.Cond = QC_ILikeLeft;
		qString = Mid(qString, 6);
	}
	else if (InStr(qString, "ILIKE") == 0)
	{
		uQCond.Cond = QC_ILikeBoth;
		qString = Mid(qString, 5);
	}
	else
	{
		log("ERROR: Query error: no valid WHERE operator found.", default.DatabaseName);
		uQCond.Cond = QC_NoCond;
		return uQCond;
	}
	Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
	
	if (InStr(qString, "?") != 0)
	{
		log("ERROR: Query syntax error: no valid condition parameter ? found.", default.DatabaseName);
		uQCond.Cond = QC_NoCond;
		return uQCond;
	}
	qString = Mid(qString, 1);
	Class'uUtils'.static.stripLeadingWhiteSpaces(qString);
	
	return uQCond;
}

final static function uQueryDump(uQuery uQ)
{
local int i;

	log("~~~~ uQuery DUMP: "$String(default.DatabaseName)$" ~~~~~", default.DatabaseName);
	log(" <> Operation = "$default.EQOpNames[uQ.Operation], default.DatabaseName);
	log(" <> Table = "$uQ.Table, default.DatabaseName);
	if (uQ.Table != None)
	{
		log(" <> Return Cols (Size) = "$uQ.returnColsSize, default.DatabaseName);
		for (i = 0; i < uQ.returnColsSize; i++)
			log("     "$i$" => "$uQ.Table.static.getColsName(uQ.returnCols[i]), default.DatabaseName);
			
		log(" <> AND Conditions (Size) = "$uQ.WhereAndCondsSize, default.DatabaseName);
		for (i = 0; i < uQ.WhereAndCondsSize; i++)
		{
			log("     "$i$" => "$
				uQ.Table.static.getColsName(uQ.WhereAndConds[i].colIndex)@
				default.EQCondNames[uQ.WhereAndConds[i].Cond]@"q["$uQ.WhereAndConds[i].paramIndex$"]", 
				default.DatabaseName);
		}
		
		log(" <> OR Conditions (Size) = "$uQ.WhereOrCondsSize, default.DatabaseName);
		for (i = 0; i < uQ.WhereOrCondsSize; i++)
		{
			log("     "$i$" => "$
				uQ.Table.static.getColsName(uQ.WhereOrConds[i].Cond1.colIndex)@
				default.EQCondNames[uQ.WhereOrConds[i].Cond1.Cond]@"q["$uQ.WhereOrConds[i].Cond1.paramIndex$"]"$
				" OR "$
				uQ.Table.static.getColsName(uQ.WhereOrConds[i].Cond2.colIndex)@
				default.EQCondNames[uQ.WhereOrConds[i].Cond2.Cond]@"q["$uQ.WhereOrConds[i].Cond2.paramIndex$"]", 
				default.DatabaseName);
		}
	}
	log("~~~~ uQuery DUMP End ~~~~~", default.DatabaseName);
}

defaultproperties
{
	DatabaseName=uDBS
	TablesList(0)=(TableClassName="uDBSTable",TableName="MyTable")
	
	EQOpNames(0)="NOOP"
	EQOpNames(1)="SELECT"
	EQOpNames(2)="COUNT"
	EQOpNames(3)="SUM"
	EQOpNames(4)="MAX"
	EQOpNames(5)="MIN"
	EQOpNames(6)="AVG"
	EQOpNames(7)="UPDATE"
	EQOpNames(8)="INSERT"
	EQOpNames(9)="DELETE"
	
	
	EQCondNames(0)="NOCOND"
	EQCondNames(1)="="
	EQCondNames(2)="<>"
	EQCondNames(3)=">="
	EQCondNames(4)=">"
	EQCondNames(5)="<="
	EQCondNames(6)="<"
	EQCondNames(7)="likeR"
	EQCondNames(8)="likeL"
	EQCondNames(9)="like"
	EQCondNames(10)="iLikeR"
	EQCondNames(11)="iLikeL"
	EQCondNames(12)="iLike"
}


