////////////////////////////////////////////////////////////////////////////////
//	PheonixMut
//
//	created by :[lol]:Mhor of www.teamlol.com
//
//	Captures player disconnects and preserves their score when reconnecting.
//	Intent is to keep people from using the disconnect/reconnect shuffle to
//	clear their scores. Side benefit is to preserve the scores of those who
//	happen to lag out during gameplay.
//
//	Release History:
//	1.0 - First development build.
//	1.1 - First beta release.
//	1.2 - Second beta release, added logging for caught players.
//	2.0 - First official release, added:
//			~Mode selection: name only, IP only, either, both
//			~When to disable: game ended, overtime, before click-in, when player
//				score reaches certain level, time before game ended, etc.
//			~Method to deactivate mutator without removing from game
//	2.1 - Ammo preservation for chosen weapon type (configurable via ini)
//	2.2 - Health preservation added.
//	3.0 - Added backup arrays updated every 15 seconds to catch lagouts that
//          are not caught by the mutator at this time. This is because the
//          leftmessage for a lagout is broadcast long after the player is
//          actually gone.
//	3.1 - Fixed a malfunctioning code check on the PlayerPawn's state to see if
//          they were dead and waiting to respawn. Settled on checking for
//          health>0, and using health as an indicator of whether to reassign
//          ammo when caught re-entering the server.
//	3.2 - Added an update info call in MutatorTakeDamage() for more frequent
//          updates; health and ammo weren't being caught as frequently as
//          needed.
//	3.3 - Updated by [ACE]CPan to fix the Gone Player update call due to the
//          change of the LeaveMessage by the ASC created by MSull
//          ( ASC Web Site: http://www.atomicunreal.com/asc/ )
//
////////////////////////////////////////////////////////////////////////////////
class PheonixMut extends Mutator config;

//use these three strings for logging and display
var	string	sName, sVersion, sCreator;

var	bool	Initialized;	//uses to initialize mutator only once

var	string	QuitMsg;		//the broadcast message when someone leaves the game
var	int		QuitMsgLen;		//length of QuitMsg

var	string	GoneName[32];		//list of disconnected players
var	float	GoneScore[32];		//corresponding scores
var	float	GoneDeaths[32];		//corresponding deaths
var	string	GoneIP[32];			//corresponding IP addy
var	int		GoneAmmo[32];		//corresponding main weapon ammo
var	int		GoneHealth[32];		//corresponding health

var	string	StoreName[32];		//list of stored playernames
var	float	StoreScore[32];		//corresponding scores
var	float	StoreDeaths[32];	//corresponding deaths
var	string	StoreIP[32];		//corresponding IP addy
var	int		StoreAmmo[32];		//corresponding main weapon ammo
var	int		StoreHealth[32];	//corresponding health

// First backup array
var	string	B1Name[32];		//list of stored playernames
var	float	B1Score[32];	//corresponding scores
var	float	B1Deaths[32];	//corresponding deaths
var	string	B1IP[32];		//corresponding IP addy
var	int		B1Ammo[32];		//corresponding main weapon ammo
var	int		B1Health[32];	//corresponding health

// Second backup array
var	string	B2Name[32];		//list of stored playernames
var	float	B2Score[32];	//corresponding scores
var	float	B2Deaths[32];	//corresponding deaths
var	string	B2IP[32];		//corresponding IP addy
var	int		B2Ammo[32];		//corresponding main weapon ammo
var	int		B2Health[32];	//corresponding health

var config	int		Mode;		//Mode of operation
var	config	bool	Active, bUseASC;		//Is mutator active
var	config	string	CheckWeapon;	//Weapon class to check the ammo
var	bool	WeaponValid;		//Are we using CheckWeapon

var	class<Weapon>	WeaponClass;	//The dynamically loaded class for our weap

function PostBeginPlay()
{
	//Keep our mutator from being registered multiple times.
	if(Initialized)
		return;
	Initialized = True;

	if(Mode!=0 && Mode!=1 && Mode!=2 && Mode!=3)
		Mode=0;

	if(Active)
	{
		//Grab some basic info about the player left message.
		//If not using the ASC leave this false to use the default
		//quit message.  If using the ASC, use the string message
		//as the quit message since the ASC changes the default one.
		if ( !bUseASC )
			QuitMsg=Level.Game.LeftMessage;
		else
			QuitMsg=" has disconnected from the server";

		QuitMsgLen=Len(QuitMsg);

		//Register as a message mutator, as we'll be using message monitoring
		//to perform some of our code. If not registered, then many message
		//events will not be passed to our mutator.
		Level.Game.RegisterMessageMutator(self);

		//Register as a damage mutator, as we'll be using damage checks to
		//update the stored information (new in v3.2).
		Level.Game.RegisterDamageMutator(self);

		//This timer will periodically capture players' scores, etc as backup
		//to the normal method of updating (lagouts take too long to be caught
		//by the normal method).
		SetTimer(15,True);

		//Load the weapon class whos ammo we want to check.
		if(CheckWeapon!="none")
		{
			WeaponClass=class<Weapon>(DynamicLoadObject(CheckWeapon,class'Class'));
			WeaponValid=True;
		}

		//Record that our mutator is in use, shows up in the log file
		Log("     ############################################################");
		Log("     ## "$sName$" Loading");
		Log("     ## Version, Contact: "$sVersion$", "$sCreator);
		Log("     ## Mode, Active:     "$Mode$", "$Active);
		Log("     ############################################################");
	}
}

//Called every time a player spawns. Here's where we check to see if the spawner
//is also contained in the Gone array, indicating it's a reconnect event. Also,
//update the stored information to make sure we capture as many score changing
//events as possible that aren't score kill related (bunnyfoofoo comes to mind).
function ModifyPlayer(Pawn Other)
{
	local	int		i, j;
	local	string	IP;

	//Since the weapon swappers use ModifyPlayer() to give special weapons,
	//we need to make sure our ammo reset is done AFTER that weapon is given
	//to the player, so call other mutators BEFORE our code.
	super.ModifyPlayer(Other);

	if( Active && !Level.Game.bGameEnded && (Other)!=none && Other.bIsPlayer
	&& !Other.IsA('Spectator') && !Other.IsA('Bot')
	&& Other.PlayerReplicationInfo!=none
	&& Other.PlayerReplicationInfo.PlayerName!="Player" )
	{
		IP=PlayerPawn(Other).GetPlayerNetworkAddress();
		j=InStr(IP,":");
		if( j!=-1 )
			IP=Left(IP,j);

		switch(Mode)
		{
			case 0:
				for(i=0; i<32 && GoneName[i]!=""; i++)
				{
					if(Other.PlayerReplicationInfo.PlayerName~=GoneName[i])
					{
						Log("  ## "$sName$" - Caught by name "$Other.PlayerReplicationInfo.PlayerName);
						RiseFromTheAshes(Other,i);
						break;
					}
				}
				break;

			case 1:
				for(i=0; i<32 && GoneName[i]!=""; i++)
				{
					if(IP==GoneIP[i] && IP!="")
					{
						Log("  ## "$sName$" - Caught by IP "$Other.PlayerReplicationInfo.PlayerName$"@"$IP);
						RiseFromTheAshes(Other,i);
						break;
					}
				}
				break;

			case 2:
				for(i=0; i<32 && GoneName[i]!=""; i++)
				{
					if( Other.PlayerReplicationInfo.PlayerName~=GoneName[i] || (IP==GoneIP[i] && IP!="") )
					{
						Log("  ## "$sName$" - Caught by name or IP "$Other.PlayerReplicationInfo.PlayerName$"@"$IP);
						RiseFromTheAshes(Other,i);
						break;
					}
				}
				break;

			case 3:
				for(i=0; i<32 && GoneName[i]!=""; i++)
				{
					if( Other.PlayerReplicationInfo.PlayerName~=GoneName[i] && IP==GoneIP[i] && IP!="" )
					{
						Log("  ## "$sName$" - Caught by name and IP "$Other.PlayerReplicationInfo.PlayerName$"@"$IP);
						RiseFromTheAshes(Other,i);
						break;
					}
				}
				break;
		}
	}

	if(Active)
		UpdateInfo();
}

function MutatorTakeDamage( out int ActualDamage, Pawn Victim, Pawn InstigatedBy, out Vector HitLocation, out Vector Momentum, name DamageType)
{
	super.MutatorTakeDamage(ActualDamage,Victim,InstigatedBy,HitLocation,Momentum,DamageType);

	if(Active)
		UpdateInfo();
}

//Use this function to re-assign scores (etc). Cleans up a lot of junk compared
//to having this code block repeated in every case in the ModifyPlayer() function.
function RiseFromTheAshes(Pawn MyPawn, int MyInt)
{
	MyPawn.PlayerReplicationInfo.Score=GoneScore[MyInt];
	MyPawn.PlayerReplicationInfo.Deaths=GoneDeaths[MyInt];
	if(GoneHealth[MyInt]>0)
		MyPawn.Health=GoneHealth[MyInt];
	if(MyPawn.FindInventoryType(WeaponClass)!= none && WeaponValid && GoneAmmo[MyInt]!=0 && GoneHealth[MyInt]>0)
		Weapon(MyPawn.FindInventoryType(WeaponClass)).AmmoType.AmmoAmount=GoneAmmo[MyInt];
	CleanGone(MyInt);
}

//Use this function to clean out entries from the gone arrays when a player
//reenters and is caught by Pheonix.
function CleanGone(int CI)
{
	local	int		i;

	for(i=CI; i<32 && GoneName[i]!=""; i++)
	{
		if(i==31)
		{
			GoneName[i]="";
			GoneScore[i]=0;
			GoneDeaths[i]=0;
			GoneIP[i]="";
			GoneAmmo[i]=0;
			GoneHealth[i]=0;
			break;
		}

		GoneName[i]=GoneName[i+1];
		GoneScore[i]=GoneScore[i+1];
		GoneDeaths[i]=GoneDeaths[i+1];
		GoneIP[i]=GoneIP[i+1];
		GoneAmmo[i]=GoneAmmo[i+1];
		GoneHealth[i]=GoneHealth[i+1];
	}
}

function bool MutatorBroadcastMessage( Actor Sender, Pawn Receiver, out coerce string Msg, optional bool bBeep, out optional name Type )
{
	local	string	quitter;
	local	int		i,j;
	local	bool	matched;

	if(Active)
	{
		//Thanks to the WebChatLog mutator for the Reciever.NextPawn==none check
		if(Receiver != none && Receiver.NextPawn == none && !Level.Game.bGameEnded)  // prevent duplicate messages
		{
			if(Right(Msg,QuitMsgLen)==QuitMsg)
			{
				quitter=left(Msg,Len(Msg)-QuitMsgLen);	//strips out the playername

				for(i=0; i<32 && StoreName[i]!=""; i++)
				{
					if(StoreName[i]~=quitter)
					{
						matched=true;	//found our player match

						for(j=0; j<32; j++)
						{
							if(GoneName[j]=="")
							{
								GoneName[j]=StoreName[i];
								GoneScore[j]=StoreScore[i];
								GoneDeaths[j]=StoreDeaths[i];
								GoneIP[j]=StoreIP[i];
								GoneAmmo[j]=StoreAmmo[i];
								GoneHealth[j]=StoreHealth[i];
								break;
							}

							if(j==31)
							{
								log("  ## "$sName$" - Gone Array is full");
								break;
							}
						}

						break;
					}
				}

				//if the  player wasn't caught in the main store array, check backup 1
				if(!matched)
				{
					for(i=0; i<32 && B1Name[i]!=""; i++)
					{
						if(B1Name[i]~=quitter)
						{
							matched=true;	//found our player match

							for(j=0; j<32; j++)
							{
								if(GoneName[j]=="")
								{
									GoneName[j]=B1Name[i];
									GoneScore[j]=B1Score[i];
									GoneDeaths[j]=B1Deaths[i];
									GoneIP[j]=B1IP[i];
									GoneAmmo[j]=B1Ammo[i];
									GoneHealth[j]=B1Health[i];
									break;
								}

								if(j==31)
								{
									log("  ## "$sName$" - Gone Array is full");
									break;
								}
							}

							break;
						}
					}
				}

				//if the  player wasn't caught in the backup 1 array, check backup 2
				if(!matched)
				{
					for(i=0; i<32 && B2Name[i]!=""; i++)
					{
						if(B2Name[i]~=quitter)
						{
							for(j=0; j<32; j++)
							{
								if(GoneName[j]=="")
								{
									GoneName[j]=B2Name[i];
									GoneScore[j]=B2Score[i];
									GoneDeaths[j]=B2Deaths[i];
									GoneIP[j]=B2IP[i];
									GoneAmmo[j]=B2Ammo[i];
									GoneHealth[j]=B2Health[i];
									break;
								}

								if(j==31)
								{
									log("  ## "$sName$" - Gone Array is full");
									break;
								}
							}

							break;
						}
					}
				}
			}
		}
	}

	if( NextMessageMutator != none )
	{
		//If there are other mutators monitoring messages, make sure we ask them
		//whether to allow the message to be broadcast (i.e. return their value).
		return NextMessageMutator.MutatorBroadcastMessage( Sender, Receiver, Msg, bBeep, Type );
	}

	//Else, we'll return true (true will allow the message to be broadcast).
	return true;
}

//Just another event to use as an update point for our store array (the more
//update events, the more up to date the store array will be).
function ScoreKill(pawn Killer, pawn Other)
{
	super.ScoreKill(Killer, Other);

	if(Active)
	{
		UpdateInfo();
	}
}

function UpdateInfo()
{
	local	int		i,j,k;
	local	string	IP;
	local	Pawn	P;

	//clear the previous name values, so we don't double register any players
	for(k=0; k<32; k++)
	{
		StoreName[k]="";
	}

	for(P=Level.PawnList; P!=none; P=P.nextPawn)
	{
		if( PlayerPawn(P)!=none && P.bIsPlayer && !P.IsA('Spectator') && !P.IsA('Bot')
		&& P.PlayerReplicationInfo!=none && P.PlayerReplicationInfo.PlayerName!="Player"
		&& (P.PlayerReplicationInfo.Score!=0 || P.PlayerReplicationInfo.Deaths!=0) )
		{
			IP=PlayerPawn(P).GetPlayerNetworkAddress();
			if( IP!="" )
			{
				j=InStr(IP,":");
				if( j!=-1 )
					IP=Left(IP,j);
			}
			StoreName[i]=P.PlayerReplicationInfo.PlayerName;
			StoreScore[i]=P.PlayerReplicationInfo.Score;
			StoreDeaths[i]=P.PlayerReplicationInfo.Deaths;
			StoreIP[i]=IP;

			if(P.Health>0)
			{
				StoreHealth[i]=P.Health;
			}

			if(P.FindInventoryType(WeaponClass)!= none && WeaponValid)
			{
				if(Weapon(P.FindInventoryType(WeaponClass)).AmmoType.AmmoAmount!=0)
					StoreAmmo[i]=Weapon(P.FindInventoryType(WeaponClass)).AmmoType.AmmoAmount;
				else
					StoreAmmo[i]=1;	//give a bullet to be nice :)
			}

			i++;
		}
	}
}

function Timer()
{
	local	int		i,j,k;
	local	string	IP;
	local	Pawn	P;

	//Shuffle backup 1 val's into the backup 2 array and clear the backup 1 array.
	//It's enough to clear just the names, as that is what is checked in the for
	//loops above.
	for(k=0; k<32; k++)
	{
		B2Name[k]=B1Name[k];
		B2Score[k]=B1Score[k];
		B2Deaths[k]=B1Deaths[k];
		B2IP[k]=B1IP[k];
		B2Ammo[k]=B1Ammo[k];
		B2Health[k]=B1Health[k];

		B1Name[k]="";
	}

	for(P=Level.PawnList; P!=none; P=P.nextPawn)
	{
		if( PlayerPawn(P)!=none && P.bIsPlayer && !P.IsA('Spectator') && !P.IsA('Bot')
		&& P.PlayerReplicationInfo!=none && P.PlayerReplicationInfo.PlayerName!="Player"
		&& (P.PlayerReplicationInfo.Score!=0 || P.PlayerReplicationInfo.Deaths!=0) )
		{
			IP=PlayerPawn(P).GetPlayerNetworkAddress();
			if( IP!="" )
			{
				j=InStr(IP,":");
				if( j!=-1 )
					IP=Left(IP,j);
			}
			B1Name[i]=P.PlayerReplicationInfo.PlayerName;
			B1Score[i]=P.PlayerReplicationInfo.Score;
			B1Deaths[i]=P.PlayerReplicationInfo.Deaths;
			B1IP[i]=IP;

			if(P.Health>0)
				B1Health[i]=P.Health;

			if(P.FindInventoryType(WeaponClass)!= none && WeaponValid)
			{
				if(Weapon(P.FindInventoryType(WeaponClass)).AmmoType.AmmoAmount!=0)
					B1Ammo[i]=Weapon(P.FindInventoryType(WeaponClass)).AmmoType.AmmoAmount;
				else
					B1Ammo[i]=1;	//give a bullet to be nice :)
			}

			i++;
		}
	}
}

defaultproperties
{
    sName="Pheonix"
    sVersion="3.1"
    sCreator="Clan :[lol]: www.teamlol.com"
    Active=True
    CheckWeapon="none"
    bUseASC=False
}
