/////////////////////////////////////////////////////////////////////////////
// iCore.h

#ifndef _ICORE_H_
#define _ICORE_H_

// Build options.
#ifdef FINAL_RELEASE
#define DO_CHECK	0
#define STATS		0
#else
#define DO_CHECK	1
#define STATS		1
#endif
#define DEMOVERSION	0

#define SUPPORTS_PRAGMA_PACK 1
#include "iVcWin32.h"

// CPU specific includes.
#if ((__INTEL__) && (!defined __GNUC__))
#define __HAS_SSE__ 1
#pragma warning(disable : 4799)
#include <xmmintrin.h>
#include <fvec.h>
#endif

/////////////////////////////////////////////////////////////////////////////
// Global constants.

enum {MAXBYTE		= 0xff       };
enum {MAXWORD		= 0xffffU    };
enum {MAXDWORD		= 0xffffffffU};
enum {MAXSBYTE		= 0x7f       };
enum {MAXSWORD		= 0x7fff     };
enum {MAXINT		= 0x7fffffff };
enum {INDEX_NONE	= -1         };
enum {UNICODE_BOM   = 0xfeff     };
enum ENoInit {E_NoInit = 0};

// Unicode or single byte character set mappings.
#ifdef _UNICODE
#ifndef _TCHAR_DEFINED
typedef UNICHAR  TCHAR;
#endif
typedef UNICHARU TCHARU;
#ifndef _TEXT_DEFINED
#undef TEXT
#define TEXT(s) L##s
#endif
#ifndef _US_DEFINED
#undef US
#define US FString(L"")
#endif

inline TCHAR FromAnsi( ANSICHAR In ) { return (BYTE)In; }
inline TCHAR FromUnicode( UNICHAR In ) { return In; }
inline ANSICHAR ToAnsi ( TCHAR In ) { return (_WORD)In<0x100 ? In : MAXSBYTE; }
inline UNICHAR ToUnicode ( TCHAR In ) { return In; }
#else
#ifndef _TCHAR_DEFINED
typedef ANSICHAR TCHAR;
#endif
typedef ANSICHARU TCHARU;

#undef TEXT
#define TEXT(s) s
#undef US
#define US FString("")

inline TCHAR FromAnsi ( ANSICHAR In ) { return In; }
inline TCHAR FromUnicode( UNICHAR In ) { return (_WORD)In<0x100 ? In : MAXSBYTE; }
inline ANSICHAR ToAnsi ( TCHAR In ) { return (_WORD)In<0x100 ? In : MAXSBYTE; }
inline UNICHAR ToUnicode ( TCHAR In ) { return (BYTE)In; }
#endif

/////////////////////////////////////////////////////////////////////////////
// Objects.

class UEnum;
class UProperty;
class UFunction;
class USystem;

// Structs.
class FArchive;
class FMemStack;
class FArray;
class FLazyLoader;
class FString;
class FMalloc;

// Templates.
template<class T> class TArray;
template<class T> class TLazyArray;
template<class TK, class TI> class TMap;
template<class TK, class TI> class TMultiMap;

// Globals.
extern class FOutputDevice* GNull;

#include "iNames.h"

// Ugly VarArgs type checking (debug builds only).
#define VARARG_EXTRA(A) A,
#define VARARG_NONE
#define VARARG_PURE =0

#define VARARG_DECL( FuncRet, StaticFuncRet, Return, FuncName, Pure, FmtType, ExtraDecl, ExtraCall )	\
	FuncRet FuncName( ExtraDecl FmtType Fmt, ... ) Pure
#define VARARG_BODY( FuncRet, FuncName, FmtType, ExtraDecl )		\
	FuncRet FuncName( ExtraDecl FmtType Fmt, ... )

//
// Abstract interfaces.
//
class FOutputDevice
{
public:
	// FOutputDevice interface.
	virtual void Serialize( const TCHAR* V, EName Event )=0;
	virtual void Flush(){};

	/**
	 * Closes output device and cleans up. This can't happen in the destructor
	 * as we might have to call "delete" which cannot be done for static/ global
	 * objects.
	 */
	virtual void TearDown(){};
		
	// Simple text printing.
	void Log( const TCHAR* S );
	void Log( enum EName Type, const TCHAR* S );
	void Log( const FString& S );
	void Log( enum EName Type, const FString& S );
	VARARG_DECL( void, void, {}, Logf, VARARG_NONE, const TCHAR*, VARARG_NONE, VARARG_NONE );
	VARARG_DECL( void, void, {}, Logf, VARARG_NONE, const TCHAR*, VARARG_EXTRA(enum EName E), VARARG_EXTRA(E) );
};

/**
 * Abstract base version of FOutputDeviceRedirector, needed due to order of dependencies.
 */
class FOutputDeviceRedirectorBase : public FOutputDevice
{
public:
	/**
	 * Adds an output device to the chain of redirections.	
	 * @param OutputDevice	output device to add
	 */
	virtual void AddOutputDevice( FOutputDevice* OutputDevice ) = 0;
	/**
	 * Removes an output device from the chain of redirections.	
	 * @param OutputDevice	output device to remove
	 */
	virtual void RemoveOutputDevice( FOutputDevice* OutputDevice ) = 0;
	/**
	 * Returns whether an output device is currently in the list of redirectors.
	 * @param	OutputDevice	output device to check the list against
	 * @return	TRUE if messages are currently redirected to the the passed in output device, FALSE otherwise
	 */
	virtual UBOOL IsRedirectingTo( FOutputDevice* OutputDevice ) = 0;
};

// Error device.
class FOutputDeviceError : public FOutputDevice
{
public:
	virtual void HandleError()=0;
};

/**
 * This class servers as the base class for console window output.
 */
class FOutputDeviceConsole : public FOutputDevice
{
public:
	/**
	 * Shows or hides the console window. 
	 * @param ShowWindow Whether to show (TRUE) or hide (FALSE) the console window.
	 */
	virtual void Show( UBOOL ShowWindow )=0;

	/** 
	 * Returns whether console is currently shown or not
	 * @return TRUE if console is shown, FALSE otherwise
	 */
	virtual UBOOL IsShown()=0;
};

// Memory allocator.
enum
{
	MEMORYIMAGE_NotUsed		= 0,
	MEMORYIMAGE_Loading		= 1,
	MEMORYIMAGE_Saving		= 2,
	MEMORYIMAGE_DebugXMemAlloc	= 4,
};

enum ECacheBehaviour
{
	CACHE_Normal		= 0,
	CACHE_WriteCombine	= 1,
	CACHE_None			= 2
};

class FMalloc
{
public:
	virtual void* Malloc( DWORD Count ) { return NULL; }
	virtual void* Realloc( void* Original, DWORD Count ) { return NULL; }
	virtual void Free( void* Original ) {}
	virtual void* PhysicalAlloc( DWORD Count, ECacheBehaviour CacheBehaviour = CACHE_WriteCombine ) { return NULL; }
	virtual void PhysicalFree( void* Original ) {}
	virtual void DumpAllocs() {}
	virtual void HeapCheck() {}
	virtual void Init( UBOOL Reset ) {}
	virtual void Exit() {}
	virtual void DumpMemoryImage() {}
	/**
	 * Gathers memory allocations for both virtual and physical allocations.
	 * @param Virtual	[out] size of virtual allocations
	 * @param Physical	[out] size of physical allocations	
	 */
	virtual void GetAllocationInfo( SIZE_T& Virtual, SIZE_T& Physical ) { Virtual = Physical = 0; }
};

// Any object that is capable of taking commands.
class FExec
{
public:
	virtual UBOOL Exec( const TCHAR* Cmd, FOutputDevice& Ar )=0;
};

// Notification hook.
class FNotifyHook
{
public:
	virtual void NotifyDestroy( void* Src ) {}
	virtual void NotifyPreChange( void* Src, UProperty* PropertyAboutToChange ) {}
	virtual void NotifyPostChange( void* Src, UProperty* PropertyThatChanged ) {}
	virtual void NotifyExec( void* Src, const TCHAR* Cmd ) {}
};

// Interface for returning a context string.
class FContextSupplier
{
public:
	virtual FString GetContext()=0;
};

// A context for displaying modal warning messages.
class FFeedbackContext : public FOutputDevice
{
public:
	VARARG_DECL( virtual UBOOL, UBOOL, return, YesNof, VARARG_PURE, const TCHAR*, VARARG_NONE, VARARG_NONE );
	virtual void BeginSlowTask( const TCHAR* Task, UBOOL StatusWindow )=0;
	virtual void EndSlowTask()=0;
	VARARG_DECL( virtual UBOOL, UBOOL VARARGS, return, StatusUpdatef, VARARG_PURE, const TCHAR*, VARARG_EXTRA(INT Numerator) VARARG_EXTRA(INT Denominator), VARARG_EXTRA(Numerator) VARARG_EXTRA(Denominator) );

	virtual void SetContext( FContextSupplier* InSupplier )=0;
	virtual void MapCheck_Show() {};
	virtual void MapCheck_ShowConditionally() {};
	virtual void MapCheck_Hide() {};
	virtual void MapCheck_Clear() {};
	virtual void MapCheck_Add( INT InType, void* InActor, const TCHAR* InMessage ) {};

	INT WarningCount;
	INT ErrorCount;
	UBOOL TreatWarningsAsErrors;

	/**
	 * A pointer to the editors frame window.  This gives you the ability to parent windows
	 * correctly in projects that are at a lower level than UnrealEd.
	 */
	DWORD	winEditorFrame;				

	FFeedbackContext()
		: ErrorCount( 0 )
		, WarningCount( 0 )
		, TreatWarningsAsErrors( 0 )
		, winEditorFrame( 0 )
    {}
};

// Class for handling undo/redo transactions among objects.
typedef void( *STRUCT_AR )( FArchive& Ar, void* TPtr );
typedef void( *STRUCT_DTOR )( void* TPtr );
class FTransactionBase
{
public:
	virtual void Apply()=0;
};

// File manager.
enum EFileTimes
{
	FILETIME_Create = 0,
	FILETIME_LastAccess = 1,
	FILETIME_LastWrite	 = 2,
};
enum EFileWrite
{
	FILEWRITE_NoFail            = 0x01,
	FILEWRITE_NoReplaceExisting = 0x02,
	FILEWRITE_EvenIfReadOnly    = 0x04,
	FILEWRITE_Unbuffered        = 0x08,
	FILEWRITE_Append			= 0x10,
	FILEWRITE_AllowRead         = 0x20,
};
enum EFileRead
{
	FILEREAD_NoFail = 0x01,
};
enum ECopyCompress
{
	FILECOPY_Normal		= 0x00,
	FILECOPY_Compress	= 0x01,
	FILECOPY_Decompress	= 0x02,
};
enum ECopyResult
{
	COPY_OK			= 0x00,
	COPY_MiscFail		= 0x01,
	COPY_ReadFail		= 0x02,
	COPY_WriteFail		= 0x03,
	COPY_CompFail		= 0x04,
	COPY_DecompFail		= 0x05,
	COPY_Canceled		= 0x06,
};
//#define COMPRESSED_EXTENSION	TEXT(".uz2")

struct FCopyProgress
{
	virtual UBOOL Poll( FLOAT Fraction )=0;
};

const TCHAR* appBaseDir();

class FFileManager
{
public:
	virtual void Init(UBOOL Startup) {}
	virtual FArchive* CreateFileReader( const TCHAR* Filename, DWORD ReadFlags=0, FOutputDevice* Error=GNull )=0;
	virtual FArchive* CreateFileWriter( const TCHAR* Filename, DWORD WriteFlags=0, FOutputDevice* Error=GNull )=0;
	virtual INT FileSize( const TCHAR* Filename )=0;
	virtual UBOOL IsReadOnly( const TCHAR* Filename )=0;
	virtual UBOOL Delete( const TCHAR* Filename, UBOOL RequireExists=0, UBOOL EvenReadOnly=0 )=0;
	virtual DWORD Copy( const TCHAR* Dest, const TCHAR* Src, UBOOL Replace=1, UBOOL EvenIfReadOnly=0, UBOOL Attributes=0, DWORD Compress=FILECOPY_Normal, FCopyProgress* Progress=NULL )=0;
	virtual UBOOL Move( const TCHAR* Dest, const TCHAR* Src, UBOOL Replace=1, UBOOL EvenIfReadOnly=0, UBOOL Attributes=0 )=0;
	virtual UBOOL MakeDirectory( const TCHAR* Path, UBOOL Tree=0 )=0;
	virtual UBOOL DeleteDirectory( const TCHAR* Path, UBOOL RequireExists=0, UBOOL Tree=0 )=0;
	virtual void FindFiles( TArray<FString>& FileNames, const TCHAR* Filename, UBOOL Files, UBOOL Directories )=0;
	virtual DOUBLE GetFileAgeSeconds( const TCHAR* Filename )=0;
	virtual UBOOL SetDefaultDirectory()=0;
	virtual FString GetCurrentDirectory()=0;
};

//
// Global variables.
//

class FOutputDeviceRedirector;

extern FMemStack					GMem;
extern FOutputDeviceRedirectorBase*	GLog;
extern FOutputDevice*				GNull;
extern FOutputDevice*				GThrow;
extern FOutputDeviceError*			GError;
extern FFeedbackContext*			GWarn;
extern FOutputDeviceConsole*		GLogConsole;
extern FMalloc*						GMalloc;
extern FFileManager*				GFileManager;
extern TCHAR						GErrorHist[4096];
extern TCHAR			GTrue[64], GFalse[64], GYes[64], GNo[64], GNone[64];
extern DOUBLE			GSecondsPerCycle;
extern DWORD			GPageSize;
extern UBOOL			GIsCriticalError;
extern UBOOL			GIsStarted;
extern UBOOL			GIsRunning;
extern UBOOL			GIsSlowTask;
extern UBOOL			GIsRequestingExit;
extern UBOOL			GUnicode;
extern UBOOL			GUnicodeOS;
extern class FArchive*	GDummySave;
extern DOUBLE			GCurrentTime;
extern class FCriticalSection*	GHostByNameCriticalSection;

// Normal includes.
#include "iFile.h"				// Low level utility code.
#include "iArc.h"				// Archive class.
#include "iTemplate.h"			// Dynamic arrays.
#include "iMem.h"				// Stack based memory management.
#include "iThreadingBase.h"		// Non-platform specific multi-threaded support.
#include "iOutputRedirector.h"	// Output redirector.

extern "C" TCHAR GLogname[];
extern "C" TCHAR GVersion[];
extern "C" TCHAR GEventname[];

#endif//_ICORE_H_

/////////////////////////////////////////////////////////////////////////////
// <<eof>> Core.h
/////////////////////////////////////////////////////////////////////////////
