// Stack container for Unreal Tournament 4.36 by James P.
// LIFO (Last In First Out) version
// This can be used for storing objects for later use
// Inspired by http://www.cs.bu.edu/teaching/cs112/spring-2000/stack/
// Send comments and suggestions to iamfearless@gmail.com

class RecyclingFIFOStack extends FIFOStack;

// The Unreal Engine 1 garbage collector is only triggered at the end of a game
// so this is used to store instances for reuse
var private SEInstanceRecycler recycler;

// Add an object to the Stack
function push( Object anObject )
{
	local StackElement se, oldTopStackElement;

	se = getElement(); // get a StackElement

	if( se == none )
	{
		log( "Unable to get StackElement instance" );
		return;
	}

	if( isEmpty() ) // if stack is empty
	{
		setBottom( se );
		getBottom().setNext( none );
		getBottom().setReference( anObject );

		setTop( getBottom() ); // Bottom and top StackElement are one and the same
	}
	else
	{
		oldTopStackElement = getTop(); // Save reference to the current top StackElement
		setTop( se ); // Set topStackElement attribute to refer to the new StackElement
		getTop().setPrevious( oldTopStackElement );

		getTop().setReference( anObject ); // Store reference to given object
	}

	setSize( getSize() + 1 );
}

function private StackElement getElement()
{
	local StackElement se;

	if( recycler == none )
		recycler = new class'SEInstanceRecycler';

	if( !recycler.isEmpty() )
	{
		se = StackElement( recycler.pop() );

		if( se == none )
		{
			log( "No instance retrieved from seInstanceRecycler" );
			return none;
		}
	}
	else // No reusable StackElements found
	{
		se = new class'StackElement';

		if( se != none ) // Instatiation succesful
			setNumInstances( getNumInstances() + 1 ); // Increment instance counter
		else
		{
			log( "Insufficient memory to instantiate a StackElement" , 'Stack' );
			return none;
		}
	}

	se.initialize();

	return se;
}

function private saveElement( StackElement se )
{
	if( recycler == none )
		recycler = new class'SEInstanceRecycler';

	se.initialize();

	recycler.push( se );
}

// Get an object from the Stack (and suppress corresponding StackElement)
function Object pop()
{
	local Object anObject;
	local StackElement oldTopStackElement;

	if( isEmpty() )
		return none;

	anObject = getBottom().getReference(); // get object
	saveElement( getBottom() );

	if( getTop() == getBottom() )
	{
		setTop( none );
		setBottom( none );
	}
	else
		setBottom( getBottom().getNext() );

	setSize( getSize() - 1 );

	return anObject;
}

function SEInstanceRecycler getInstanceRecycler()
{
	return recycler;
}

function int initialize()
{
	local StackElement se;

	while( getTop() != none )
	{
		se = StackElement( pop() );

		if( se != none )
			saveElement( se );
	}

	setSize( 0 );
	setTop( none );
	setBottom( none );
}

// Test method
function logStackContents()
{
	local StackElement se;

	se = getTop();

	while( se != none )
	{
		log( se.getReference() );

		se = se.getPrevious();
	}
}

defaultproperties
{

}
