//================================================================================
// PlayerTools for Unreal Tournament 4.36 by James P.
// A class for mod authors containing some useful functions
// Please give credit to the author if you modify this mod
//================================================================================

class PlayerTools extends Object;

// Functions in this class :
// - getLastPlayer()
// - getNextPlayer()

// Special function to simulate Pickup
function givePickupToPlayer( Pawn p, string itemClass )
{
	local class<TournamentPickup> PUClass;
	local Pickup PU;

	if( itemClass == "" )
		return;

	PUClass = class<TournamentPickup>(DynamicLoadObject(ItemClass, class'Class'));

	PU = Spawn( PUClass );

	if( PU != none )
	{
		PU.Touch( P ); // Touch player
		PU.Destroy(); // Get rid of spawned pickup
	}
}

function string getPlayerIP( Pawn p )
{
	if( isValidPawn( p ) )
	{
		ip =  PlayerPawn( p ).GetPlayerNetworkAddress();
		ip = left( ip, Instr( ip, ":" ) );

		return ip;
	}

	return "0.0.0.0";
}

function bool removePlayersWeapons( Pawn p )
{
	local Inventory inv, nextInventory;

	if( isValidPawn( p ) )
	{
		inv = p.Inventory;

		while( Inv != none )
		{
			nextInventory = Inv.Inventory;

			inv.DropInventory();
			inv.Destroy();

			inv = nextInventory;
		}

		if( p.Inventory == none )
			return true;
	}

	return false;
}

function bool movePlayer( Pawn p, Vector location )
{
	// Notes:
	// - Improve this so it checks if it's possible for the player to exist in the given location
	// - Use a dud player ?

	if( p != none && location != p.Location )
	{
		p.SetCollision( false, false, false ); // Turn all collision variables off
		p.SetLocation( location ); // Move the player
		p.SetCollision( true, true, true );	// Make player solid again

		if( p.Location == location )
			return true;
		else
			return false;
	}

	return false;
}

// Swaps player locations around
// Dependencies: movePlayer()
function bool swapPlayerLocations( Pawn p1, Pawn p2 )
{
	local vector tempLocation1, tempLocation2;

	if( p1 != none && p2 != none )
	{
		// Store location of players
		tempLocation1 = p1.Location;
		tempLocation2 = p2.Location;

		movePlayer( p1, tempLocation2 ); // Move the player

		movePlayer( p2, tempLocation1 ); // Move the player

		if( p1.Location == tempLocation2 && p2.Location == tempLocation1 )
			return true;
	}

	return false;
}

function sendMessageToPlayer( Pawn p, string message )
{
	if( isValidPlayer( p ) )
		p.ClientMessage( message );
}

function int getPlayerScore( Pawn p )
{
	if( isValidPlayer( p ) )
		return p.PlayerReplicationInfo.Score;

	return -999999;
}

function int getPlayerDeaths( Pawn p )
{
	if( isValidPlayer( p ) )
		return p.DieCount;

	return -999999;
}

function int getPlayerKills( Pawn p )
{
	if( isValidPlayer( p ) )
		return p.KillCount;

	return -999999;
}

function float getPlayerFPH( Pawn p )
{
	float fph;

	if( isValidPlayer( p ) )
	{
		FPH = p.KillCount / ( ( Level.TimeSeconds - p.PlayerReplicationInfo.StartTime ) * 3600 );

		return FPH;
	}

	return -999999.9;
}

function Pawn getNextPlayer()
{
	return none;
}

function Pawn getLastPlayer()
{
	return none;
}

function Pawn getFirstPlayer()
{
	local Pawn p;

    p = Level.PawnList;

	while( p != none )
	{
		if( isValidPawn( p ) )
				return p;

        p = p.nextPawn;
	}

	return none;
}

// Dependencies: none
function int getRandomPlayerID()
{
	local int numberOfPlayers;

	if( Level.Game != none )
	{
		numberOfPlayers = DeathMatchPlus(Level.Game).NumPlayers + DeathMatchPlus(Level.Game).NumBots;

		return rand( numberOfPlayers + 1 );
	}

	return -1;
}

// Dependencies: getRandomPlayerID(), getPawnWithPID()
function Pawn getRandomPlayer()
{
	local int randomPID;
	local Pawn p;

	if( Level.Game != none )
	{
		randomPID = getRandomPlayerID();

		p = getPawnWithPID( randomPID );

		return p;
	}

	return none;
}

// Dependencies: isValidPlayer()
function Pawn getPawnWithName( string playerName )
{
	local Pawn p;

    p = Level.PawnList;

	while( p != none )
	{
		if( isValidPawn( p ) )
			if( p.PlayerReplicationInfo.PlayerName == playerName )
				return p;

        p = p.nextPawn;
	}

	return none;
}

// Dependencies: isValidPlayer()
function Pawn getPawnWithPID( int playerID )
{
    local Pawn p;

    p = Level.PawnList;

    while ( p != none )
    {
        if( isValidPawn( p ) )
            if( P.PlayerReplicationInfo.PlayerID == playerID ) // Pawn found
                return p; // So return reference to Pawn

        p = p.nextPawn;
    }

	return none; // No corresponding Pawn found
}

// Dependencies: none
function bool isValidPlayer( Pawn p )
{
	if( p != none && p.IsA('PlayerPawn') &&
		PlayerPawn( p ).PlayerReplicationInfo != none &&
		!p.IsA('Spectator') )
		return true;

	return false;
}

// Dependencies: isValidPlayer()
function bool kickPlayer( Pawn p )
{
	if( isValidPawn( p ) )
	{
		p.Destroy();

		if( p == none ) // Kick succesful
			return true;
	}

	return false;
}

// Dependencies: none
function throwPlayer( Pawn p, vector throwVector )
{
	if( isValidPlayer( p ) )
		p.AddVelocity( throwVector );
}

// Dependencies: isValidPlayer()
function bool playerWithinDistanceOfPoint( Pawn p, vector pointLocation, int distanceAmount )
{
	local float deltaDistance;

	if( isValidPlayer( p ) )
	{
		DeltaDistance = VSize( p.Location - pointLocation );

		if( deltaDistance <= distanceAmount )
			return true;
	}

	return false;
}
