//////////////////////////////////////////////////////////////////////////////
// uDBSTCol - UScript Database System Table column
//		Feralidragon - 09-07-2013
//
//		uWebDrvX 0.1
//////////////////////////////////////////////////////////////////////////////

class uDBSTCol extends Object config(uDBS)
abstract;

enum ECType
{
	CT_BOOL,
	CT_BYTE,
	CT_INT,
	CT_FLOAT,
	CT_NAME,
	CT_STRING,
	CT_CLASS,
	CT_TEXTURE,
	CT_SOUND,
	CT_MESH,
	CT_VECTOR,
	CT_ROTATOR,
	CT_COLOR
};
var const ECType ColType;

var() config int CSize;
var int loadedCSize;
var class<uDBS> uDBSClass;
var class<uDBSTable> uDBSTableClass;



static function int getRealSize();
static function bool addDefault();

final static function int getSize()
{
	return default.loadedCSize;
}

final static function byte getType()
{
	return default.ColType;
}

final static function string getHumanReadableType()
{
	switch (default.ColType)
	{
		case CT_BOOL:		return "BOOL";
		case CT_BYTE:		return "BYTE";
		case CT_INT:		return "INT";
		case CT_FLOAT:		return "FLOAT";
		case CT_NAME:		return "NAME";
		case CT_STRING:		return "STRING";
		case CT_CLASS:		return "CLASS";
		case CT_TEXTURE:	return "TEXTURE";
		case CT_SOUND:		return "SOUND";
		case CT_MESH:		return "MESH";
		case CT_VECTOR:		return "VECTOR";
		case CT_ROTATOR:	return "ROTATOR";
		case CT_COLOR:		return "COLOR";
	}
	return "";
}

final static function triggerColumnOverLimits(string traceBack, int curVal, int maxVal)
{
	log("ERROR: Column over limits '"$traceBack$"': trying to insert row "$curVal$", only "$maxVal$" max rows are allowed. "$
		"Please purge some data first.", default.uDBSClass.default.DatabaseName);
}

final static function triggerOverflowError(string traceBack, int curVal, int maxVal)
{
	log("ERROR: Row access overflow '"$traceBack$"': trying to access row "$curVal$", max row is "$maxVal$".", default.uDBSClass.default.DatabaseName);
}

static function bool loadCol(class<uDBS> DBClass, class<uDBSTable> DBTable)
{
	default.uDBSClass = DBClass;
	default.uDBSTableClass = DBTable;
	if (default.uDBSClass == None || default.uDBSTableClass == None)
		return False;
	default.loadedCSize = default.CSize;
	if (default.loadedCSize < 0 || default.loadedCSize >= getRealSize())
	{
		triggerOverflowError(default.Class$".loadCol()", default.CSize, getRealSize());
		return False;
	}
	return True;
}


static function string getToString(int index)
{
	if (index < 0 || index >= default.loadedCSize)
	{
		triggerOverflowError(default.Class$".getToString()", index, default.loadedCSize);
		return "ERROR";
	}
	return "";
}

final static function bool processValidIndex(int index, string traceBack)
{
	if (index < 0 || index >= default.loadedCSize)
	{
		triggerOverflowError(traceBack, index, default.loadedCSize);
		return False;
	}
	return True;
}

final static function bool processValidAdd(out int index, string traceBack)
{
	default.loadedCSize++;
	if (default.loadedCSize >= getRealSize())
	{
		index = -1;
		triggerColumnOverLimits(traceBack, default.loadedCSize, getRealSize());
		return False;
	}
	index = default.loadedCSize - 1;
	return True;
}


////////////////////////////////////////////////////////////////////////////////////////////////////
// GET operation
////////////////////////////////////////////////////////////////////////////////////////////////////
static function bool getBool(int index)
{
	return (default.ColType == CT_BOOL && processValidIndex(index, default.Class$".getBool()"));
}

static function byte getByte(int index)
{
	return Byte(default.ColType == CT_BYTE && processValidIndex(index, default.Class$".getByte()"));
}

static function int getInt(int index)
{
	return Int(default.ColType == CT_INT && processValidIndex(index, default.Class$".getInt()"));
}

static function float getFloat(int index)
{
	return Float(default.ColType == CT_FLOAT && processValidIndex(index, default.Class$".getFloat()"));
}

static function name getName(int index)
{
	if (default.ColType == CT_NAME && processValidIndex(index, default.Class$".getName()"))
		return '1';
	return '';
}

static function string getString(int index)
{
	if (default.ColType == CT_STRING && processValidIndex(index, default.Class$".getString()"))
		return "1";
	return "";
}

static function class getClass(int index)
{
	if (default.ColType == CT_CLASS && processValidIndex(index, default.Class$".getClass()"))
		return Class'Object';
	return None;
}

static function texture getTexture(int index)
{
	if (default.ColType == CT_TEXTURE && processValidIndex(index, default.Class$".getTexture()"))
		return Texture'DefaultTexture';
	return None;
}

static function sound getSound(int index)
{
	if (default.ColType == CT_SOUND && processValidIndex(index, default.Class$".getSound()"))
		return Sound'NewBeep';
	return None;
}

static function mesh getMesh(int index)
{
	if (default.ColType == CT_MESH && processValidIndex(index, default.Class$".getMesh()"))
		return Mesh'PlainBar';
	return None;
}

static function vector getVector(int index)
{
	if (default.ColType == CT_VECTOR && processValidIndex(index, default.Class$".getVector()"))
		return vect(1,0,0);
	return vect(0,0,0);
}

static function rotator getRotator(int index)
{
	if (default.ColType == CT_ROTATOR && processValidIndex(index, default.Class$".getRotator()"))
		return rot(16384,0,0);
	return rot(0,0,0);
}

static function color getColor(int index)
{
local color c;

	if (default.ColType == CT_COLOR && processValidIndex(index, default.Class$".getColor()"))
		c.R = 255;
	return c;
}


////////////////////////////////////////////////////////////////////////////////////////////////////
// SET operation
////////////////////////////////////////////////////////////////////////////////////////////////////
static function bool setBool(int index, bool val)
{
	return (default.ColType == CT_BOOL && processValidIndex(index, default.Class$".setBool()"));
}

static function bool setByte(int index, byte val)
{
	return (default.ColType == CT_BYTE && processValidIndex(index, default.Class$".setByte()"));
}

static function bool setInt(int index, int val)
{
	return (default.ColType == CT_INT && processValidIndex(index, default.Class$".setInt()"));
}

static function bool setFloat(int index, float val)
{
	return (default.ColType == CT_FLOAT && processValidIndex(index, default.Class$".setFloat()"));
}

static function bool setName(int index, name val)
{
	return (default.ColType == CT_NAME && processValidIndex(index, default.Class$".setName()"));
}

static function bool setString(int index, string val)
{
	return (default.ColType == CT_STRING && processValidIndex(index, default.Class$".setString()"));
}

static function bool setClass(int index, class val)
{
	return (default.ColType == CT_CLASS && processValidIndex(index, default.Class$".setClass()"));
}

static function bool setTexture(int index, texture val)
{
	return (default.ColType == CT_TEXTURE && processValidIndex(index, default.Class$".setTexture()"));
}

static function bool setSound(int index, sound val)
{
	return (default.ColType == CT_SOUND && processValidIndex(index, default.Class$".setSound()"));
}

static function bool setMesh(int index, mesh val)
{
	return (default.ColType == CT_MESH && processValidIndex(index, default.Class$".setMesh()"));
}

static function bool setVector(int index, vector val)
{
	return (default.ColType == CT_VECTOR && processValidIndex(index, default.Class$".setVector()"));
}

static function bool setRotator(int index, rotator val)
{
	return (default.ColType == CT_ROTATOR && processValidIndex(index, default.Class$".setRotator()"));
}

static function bool setColor(int index, color val)
{
	return (default.ColType == CT_COLOR && processValidIndex(index, default.Class$".setColor()"));
}


////////////////////////////////////////////////////////////////////////////////////////////////////
// ADD operation
////////////////////////////////////////////////////////////////////////////////////////////////////
static function bool addBool(bool val, optional out int index)
{
	return (default.ColType == CT_BOOL && processValidAdd(index, default.Class$".addBool()"));
}

static function bool addByte(byte val, optional out int index)
{
	return (default.ColType == CT_BYTE && processValidAdd(index, default.Class$".addByte()"));
}

static function bool addInt(int val, optional out int index)
{
	return (default.ColType == CT_INT && processValidAdd(index, default.Class$".addInt()"));
}

static function bool addFloat(float val, optional out int index)
{
	return (default.ColType == CT_FLOAT && processValidAdd(index, default.Class$".addFloat()"));
}

static function bool addName(name val, optional out int index)
{
	return (default.ColType == CT_NAME && processValidAdd(index, default.Class$".addName()"));
}

static function bool addString(string val, optional out int index)
{
	return (default.ColType == CT_STRING && processValidAdd(index, default.Class$".addString()"));
}

static function bool addClass(class val, optional out int index)
{
	return (default.ColType == CT_CLASS && processValidAdd(index, default.Class$".addClass()"));
}

static function bool addTexture(texture val, optional out int index)
{
	return (default.ColType == CT_TEXTURE && processValidAdd(index, default.Class$".addTexture()"));
}

static function bool addSound(sound val, optional out int index)
{
	return (default.ColType == CT_SOUND && processValidAdd(index, default.Class$".addSound()"));
}

static function bool addMesh(mesh val, optional out int index)
{
	return (default.ColType == CT_MESH && processValidAdd(index, default.Class$".addMesh()"));
}

static function bool addVector(vector val, optional out int index)
{
	return (default.ColType == CT_VECTOR && processValidAdd(index, default.Class$".addVector()"));
}

static function bool addRotator(rotator val, optional out int index)
{
	return (default.ColType == CT_ROTATOR && processValidAdd(index, default.Class$".addRotator()"));
}

static function bool addColor(color val, optional out int index)
{
	return (default.ColType == CT_COLOR && processValidAdd(index, default.Class$".addColor()"));
}


////////////////////////////////////////////////////////////////////////////////////////////////////
// REMOVE operation
////////////////////////////////////////////////////////////////////////////////////////////////////
static function bool removeBool(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_BOOL && processValidIndex(index, default.Class$".removeBool()"));
}

static function bool removeByte(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_BYTE && processValidIndex(index, default.Class$".removeByte()"));
}

static function bool removeInt(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_INT && processValidIndex(index, default.Class$".removeInt()"));
}

static function bool removeFloat(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_FLOAT && processValidIndex(index, default.Class$".removeFloat()"));
}

static function bool removeName(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_NAME && processValidIndex(index, default.Class$".removeName()"));
}

static function bool removeString(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_STRING && processValidIndex(index, default.Class$".removeString()"));
}

static function bool removeClass(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_CLASS && processValidIndex(index, default.Class$".removeClass()"));
}

static function bool removeTexture(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_TEXTURE && processValidIndex(index, default.Class$".removeTexture()"));
}

static function bool removeSound(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_SOUND && processValidIndex(index, default.Class$".removeSound()"));
}

static function bool removeMesh(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_MESH && processValidIndex(index, default.Class$".removeMesh()"));
}

static function bool removeVector(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_VECTOR && processValidIndex(index, default.Class$".removeVector()"));
}

static function bool removeRotator(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_ROTATOR && processValidIndex(index, default.Class$".removeRotator()"));
}

static function bool removeColor(int index, optional bool bResetOnly)
{
	return (default.ColType == CT_COLOR && processValidIndex(index, default.Class$".removeColor()"));
}


defaultproperties
{
}
