//===========================================================================================
// medToggle
// author:  Helen! www.birdieman.com/forum

// Version 2, adding more settings. The order of settings:
// [Settings] Translocator, GameSpeed, AirControl
// [Bots]     BotSkill, MinPlayers
// [Rules]    FragLimit/MaxTeamScore, TimeLimit, WeaponStay, ForceTeamBalance, FriendlyFire

// Version 2.10
//  - splitting Frag Limit and Max Team Score into their own variables in config file.
//  - added ability to set gravity.
//===========================================================================================

class medToggle expands Mutator config(medToggle);

enum ELogType
{
	LOG_info,   // 0
	LOG_error   // 1
};

// Our local definition of Game Types
enum EArena
{
    ARENA_other,       // 0
	ARENA_assault,     // 1
	ARENA_ctf,         // 2
	ARENA_dom,         // 3
	ARENA_tdm,         // 4
	ARENA_lms,         // 5
	ARENA_dm           // 6
};


// Logging
var() config string bLogInfo;
var() config string bLogErrors;

// Settings
var() config string DMTranslocator;
var() config string TDMTranslocator;
var() config string LMSTranslocator;
var() config string ASTranslocator;
var() config string CTFTranslocator;
var() config string DOMTranslocator;
var() config float cGameSpeed;
var() config float cAirControl;

// Bots
var() config int cBotSkill;
var() config int cMinPlayers;

// Rules
var() config int cFragLimit;
var() config int TDMMaxTeamScore;
var() config int CTFMaxTeamScore;
var() config int DOMMaxTeamScore;
var() config int cTimeLimit;
var() config string cWeaponStay;
var() config string cForceRespawn;
var() config string cForceTeamBalance;
var() config float cFriendlyFire;

// Other
var() config int cGravity;

// map specific array
var config string cMap[255];

// Class vars
var bool mLogInfo;
var bool mLogErrors;
var EArena mArena;
var string mMapSettings;
var bool mDifficultySet;

function PreBeginPlay()
{
    local int i;
    local string aMapName;
    local string aCurrentMapName;
    local string s;
    local bool bFound;
    local bool bUseTrans;
    local bool bWeaponStay;
    local bool bForceSpawn;
    local bool bTeamBalance;
    local int aMaxTeamScore;
    local string aScoreDescription;

    mDifficultySet = False;

    mLogInfo = isTrue(bLogInfo);
    mLogErrors = isTrue(bLogErrors);

    doLog(LOG_error, "PreBeginPlay()");

    if (!(Level.Game.IsA('DeathMatchPlus')))
    {
        doLog(LOG_error, "this mutator only supports game types that inherit from the DeathMatchPlus class.");
        return;
    }

    // Level.Game.Class returns for example "Botpack.DeathMatchPlus"
    doLog(LOG_info, "GameType="$string(Level.Game.Class));

    // Set Game Type
    SetLocalGameType();

    // VALIDATE DEFAULTS ///////////////
    cGameSpeed = ValidateGameSpeed(string(cGameSpeed));
    cAirControl = ValidateAirControl(string(cAirControl));
    cBotSkill = ValidateBotSkill(string(cBotSkill));
    cMinPlayers = ValidateMinPlayers(string(cMinPlayers));
    cFragLimit = ValidateMaxTeamScore(string(cFragLimit));
    TDMMaxTeamScore = ValidateMaxTeamScore(string(TDMMaxTeamScore));
    CTFMaxTeamScore = ValidateMaxTeamScore(string(CTFMaxTeamScore));
    DOMMaxTeamScore = ValidateMaxTeamScore(string(DOMMaxTeamScore));
    cTimeLimit = ValidateTimeLimit(string(cTimeLimit));
    bWeaponStay = isTrue(cWeaponStay);
    bForceSpawn = isTrue(cForceRespawn);
    bTeamBalance = isTrue(cForceTeamBalance);
    cFriendlyFire = ValidateFriendlyFire(string(cFriendlyFire));
    cGravity = ValidateGravity(string(cGravity));
    // END OF VALIDATE DEFAULTS ////////

    s = "";
    s = PrepMsg(s, "ASTranslocator", string(isTrue(ASTranslocator)));
    s = PrepMsg(s, "CTFTranslocator", string(isTrue(CTFTranslocator)));
    s = PrepMsg(s, "DOMTranslocator", string(isTrue(DOMTranslocator)));
    s = PrepMsg(s, "TDMTranslocator", string(isTrue(TDMTranslocator)));
    s = PrepMsg(s, "LMSTranslocator", string(isTrue(LMSTranslocator)));
    s = PrepMsg(s, "DMTranslocator", string(isTrue(DMTranslocator)));
    s = PrepMsg(s, "cGameSpeed", string(cGameSpeed));
    s = PrepMsg(s, "AirControl", string(cAirControl));
    s = "Default Settings: "$s;
    doLog(LOG_info, s);

    s = "";
    s = PrepMsg(s, "BaseSkill", string(cBotSkill));
    s = PrepMsg(s, "MinPlayers", string(cMinPlayers));
    s = "Default Bots: "$s;
    doLog(LOG_info, s);

    s = "";
    s = PrepMsg(s, "FragLimit", string(cFragLimit));
    s = PrepMsg(s, "TDMMaxTeamScore", string(TDMMaxTeamScore));
    s = PrepMsg(s, "CTFMaxTeamScore", string(CTFMaxTeamScore));
    s = PrepMsg(s, "DOMMaxTeamScore", string(DOMMaxTeamScore));
    s = PrepMsg(s, "TimeLimit", string(cTimeLimit));
    s = PrepMsg(s, "WeaponStay", string(bWeaponStay));
    s = PrepMsg(s, "ForceRespawn", string(bForceSpawn));
    s = PrepMsg(s, "ForceTeamBalance", string(bTeamBalance));
    s = PrepMsg(s, "FriendlyFire", string(cFriendlyFire));
    s = "Default Rules: "$s;
    doLog(LOG_info, s);

    s = "";
    s = PrepMsg(s, "Gravity", string(cGravity));
    s = "Default (other): "$s;
    doLog(LOG_info, s);

    // MAP NAME ///////////
    s = string(Level.Game.BaseMutator);

    if (s == "")
    {
        doLog(LOG_error, "could not determine the value of Level.Game.BaseMutator");
        return;
    }

    doLog(LOG_info, "Level.Game.BaseMutator = "$s);

    i = InStr(s, ".");
    if(i == (-1))
    {
        // hope this is the map name
        aCurrentMapName = s;
    }
    else
    {
        // Parse out the map name
        aCurrentMapName = Left(s, i);

        if (aCurrentMapName == "")
        {
            doLog(LOG_error, "The current map name could not be determined after parsing Level.Game.BaseMutator");
            return;
        }
    }
    // END OF MAP NAME ///////////


    doLog(LOG_info, "aCurrentMapName = "$aCurrentMapName);


    // More default prep in case the map name is not found.

    if(mArena == ARENA_assault)
    {
        bUseTrans = isTrue(ASTranslocator);
    }
    else if(mArena == ARENA_lms)
    {
        bUseTrans = isTrue(LMSTranslocator);
    }
    else if(mArena == ARENA_dom)
    {
        aMaxTeamScore = DOMMaxTeamScore;
        aScoreDescription = "DOMMaxTeamScore";
        bUseTrans = isTrue(DOMTranslocator);
    }
    else if(mArena == ARENA_ctf)
    {
        aMaxTeamScore = CTFMaxTeamScore;
        aScoreDescription = "CTFMaxTeamScore";
        bUseTrans = isTrue(CTFTranslocator);
    }
    else if(mArena == ARENA_tdm)
    {
        aMaxTeamScore = TDMMaxTeamScore;
        aScoreDescription = "TDMMaxTeamScore";
        bUseTrans = isTrue(TDMTranslocator);
    }
    else
    {
        //210 aMaxTeamScore = cFragLimit;
        aScoreDescription = "cFragLimit";
        bUseTrans = isTrue(DMTranslocator);
    }

    // see if aMapName is in the array
    i = 0;
    bFound = false;
    while ((!bFound) && (i < ArrayCount(cMap)))
    {
        aMapName = ExtractMapName(cMap[i]);

        bFound = (aMapName ~= aCurrentMapName);

        if(bFound)
            mMapSettings = cMap[i];
        else
            i++;
    }

    // GET MAP SPECIFIC VALUES //////////////
    if(bFound)
    {
        // The first value is the map name, not a setting, but this will strip it out for us.
        StripSetting("");

        // Translocator
        bUseTrans = isTrue(StripSetting(string(bUseTrans)));

        // GameSpeed
        s = StripSetting(string(cGameSpeed));
        cGameSpeed = ValidateGameSpeed(s);

        // AirControl
        s = StripSetting(string(cAirControl));
        cAirControl = ValidateAirControl(s);

        // BotSkill
        s = StripSetting(string(cBotSkill));
        cBotSkill = ValidateBotSkill(s);

        // MinPlayers
        s = StripSetting(string(cMinPlayers));
        cMinPlayers = ValidateMinPlayers(s);

        // MaxTeamScore
        s = StripSetting(string(aMaxTeamScore));
        aMaxTeamScore = ValidateMaxTeamScore(s);

        // TimeLimit
        s = StripSetting(string(cTimeLimit));
        cTimeLimit = ValidateTimeLimit(s);

        // WeaponStay
        s = StripSetting(string(bWeaponStay));
        bWeaponStay = isTrue(s);

        // ForceRespawn
        s = StripSetting(string(bForceSpawn));
        bForceSpawn = isTrue(s);

        // ForceTeamBalance
        s = StripSetting(string(bTeamBalance));
        bTeamBalance = isTrue(s);

        // FriendlyFire
        s = StripSetting(string(cFriendlyFire));
        cFriendlyFire = ValidateFriendlyFire(s);

        // FragLimit
        s = StripSetting(string(cFragLimit));
        cFragLimit = ValidateMaxTeamScore(s);

        // Gravity
        s = StripSetting(string(cGravity));
        cGravity = ValidateGravity(s);

        s = "";
        s = PrepMsg(s, "Translocator", string(bUseTrans));
        s = PrepMsg(s, "GameSpeed", string(cGameSpeed));
        s = PrepMsg(s, "AirControl", string(cAirControl));
        s = aCurrentMapName$" Settings: "$s;
        doLog(LOG_info, s);

        s = "";
        s = PrepMsg(s, "BaseSkill", string(cBotSkill));
        s = PrepMsg(s, "MinPlayers", string(cMinPlayers));
        s = aCurrentMapName$" Bots: "$s;
        doLog(LOG_info, s);

        s = "";
        if(isTeamGame())
            s = PrepMsg(s, aScoreDescription, string(aMaxTeamScore));
        else if(isDeathMatch())
            s = PrepMsg(s, aScoreDescription, string(cFragLimit));


        if(mArena != ARENA_assault)
        {
            s = PrepMsg(s, "TimeLimit", string(cTimeLimit));
            s = PrepMsg(s, "WeaponStay", string(bWeaponStay));
        }

        if(isDeathMatch() || isTeamGame())
            s = PrepMsg(s, "ForceRespawn", string(bForceSpawn));

        if(isTeamGame())
        {
            s = PrepMsg(s, "ForceTeamBalance", string(bTeamBalance));
            s = PrepMsg(s, "FriendlyFire", string(cFriendlyFire));
        }

        if(s != "")
        {
            s = aCurrentMapName$" Rules: "$s;
            doLog(LOG_info, s);
        }

        s = "";
        s = PrepMsg(s, "Gravity", string(cGravity));
        s = aCurrentMapName$" Other: "$s;
        doLog(LOG_info, s);
    }
    else
    {
        doLog(LOG_info, aCurrentMapName$" not in the list, so the default settings will be used.");
    }
    // END OF GET MAP SPECIFIC VALUES //////////////


    // Set the preferences!
    DeathMatchPlus(Level.Game).bUseTranslocator = bUseTrans;
    DeathMatchPlus(Level.Game).GameSpeed = cGameSpeed;
    DeathMatchPlus(Level.Game).AirControl = cAirControl;
    DeathMatchPlus(Level.Game).MinPlayers = cMinPlayers;

    if(isTeamGame())
        TeamGamePlus(Level.Game).GoalTeamScore = aMaxTeamScore;
    else if(mArena != ARENA_assault)
        DeathMatchPlus(Level.Game).FragLimit = cFragLimit; //210 aMaxTeamScore;

    if(mArena != ARENA_assault)
    {
        DeathMatchPlus(Level.Game).TimeLimit = cTimeLimit;
        DeathMatchPlus(Level.Game).bMultiWeaponStay = bWeaponStay;
    }

    if(isDeathMatch() || isTeamGame())
    {
        DeathMatchPlus(Level.Game).bForceRespawn = bForceSpawn;
    }

    if(isTeamGame())
    {
        TeamGamePlus(Level.Game).bPlayersBalanceTeams = bTeamBalance;
        TeamGamePlus(Level.Game).FriendlyFireScale = cFriendlyFire;
    }

    // Note that we can't set botSkill yet, see Tick().
    // Note that gravity is done in CheckReplacement().
    return;
}

function Tick(float Delta)
{
    /*  In order for us to assign the bot difficulty, we have to wait
        until DeathMatchPlus(Level.Game).BotConfig is not None. It 
        eventually gets assigned in DeathMatchPlus.PostBeginPlay(),
        and the soonest available function for us to access it is Tick.
    */

    if(DeathMatchPlus(Level.Game).BotConfig == None)
    {
        //doLog(LOG_info, "DeathMatchPlus(Level.Game).BotConfig == None");
        return;
    }

    //log("before - BotConfig.Difficulty="$string(DeathMatchPlus(Level.Game).BotConfig.Difficulty), 'medToggle');

    if(DeathMatchPlus(Level.Game).BotConfig.Difficulty != cBotSkill)
    {
        DeathMatchPlus(Level.Game).BotConfig.Difficulty = cBotSkill;
        doLog(LOG_info, "Tick(), assigning the bot difficulty to "$string(cBotSkill));
    }

    //log("after - BotConfig.Difficulty="$string(DeathMatchPlus(Level.Game).BotConfig.Difficulty), 'medToggle');

    Disable('Tick');
}

function SetLocalGameType()
{
    // We need to ask for class type in this order, due to the order of inheritance.
    if(Level.Game.IsA('Assault'))
    {
        mArena = ARENA_assault;
    }
    else if(Level.Game.IsA('CTFGame'))
    {
        mArena = ARENA_ctf;
    }
    else if(Level.Game.IsA('Domination'))
    {
        mArena = ARENA_dom;
    }
    else if(Level.Game.IsA('TeamGamePlus'))
    {
        mArena = ARENA_tdm;
    }
    else if(Level.Game.IsA('LastManStanding'))
    {
        mArena = ARENA_lms;
    }
    else if(Level.Game.IsA('DeathMatchPlus'))
    {
        mArena = ARENA_dm;
    }
    else
    {
        mArena = ARENA_other;
    }
}

function string PrepMsg(string s, string ASetting, string AValue)
{
    if (s != "")
        s = s$", ";

    return s$ASetting$"="$AValue;
}

function bool isTrue(string AValue)
{
    return (AValue ~= "true");
}

function bool isTeamGame()
{
    return (Level.Game.IsA('TeamGamePlus'));
}

// By this we mean literally
function bool isDeathMatch()
{
    return (mArena == ARENA_dm);
    //return ( (!isTeamGame()) && (mArena != ARENA_LMS));
}

// This returns the first setting, as well as removes it from the string.
// If the setting is not found, the ADefault will be returned.
function string StripSetting(string ADefault)
{
    local int i;
    local string s;

    if (mMapSettings == "")
        return ADefault;

    i = InStr(mMapSettings, ",");
    if(i == (-1))
    {
        s = mMapSettings;
        mMapSettings = "";
        return s;
    }

    s = Left(mMapSettings, i);
    mMapSettings = Mid(mMapSettings, i+1);
    
    if(s=="")
        return ADefault;

    return s;
}

function string ExtractMapName(string ASetting)
{
    local int i;

    if (ASetting == "")
        return "";

    i = InStr(ASetting, ",");
    if(i == (-1))
        return ASetting;

    return Left(ASetting, i);
}

function string ExtractTranslocatorSetting(string ASetting)
{
    local int i;

    if (ASetting == "")
        return "";

    i = InStr(ASetting, ",");
    if(i == (-1))
        return "";

    if ((i+1)==Len(ASetting))
        return "";

    return Mid(ASetting, i+1);
}

function doLog(ELogType ALogType, string AMessage)
{
    if(  ( (ALogType == LOG_info) && (mLogInfo) ) || ((ALogType == LOG_error) && (mLogErrors))  )
        log(AMessage, 'medToggle');
}

function float ValidateGameSpeed(string AGameSpeed)
{
    local float nGameSpeed;
    
    // If AGameSpeed is not a valid float, the result will be zero.
    nGameSpeed = float(AGameSpeed);

    if((nGameSpeed <= 0) || (nGameSpeed > 2))
        nGameSpeed = 1;

    return nGameSpeed;
}

function float ValidateAirControl(string AValue)
{
    local float nValue;

    // If AValue is not a valid float, the result of this cast will be zero.
    nValue = float(AValue);

    if((nValue < 0) || (nValue > 1))
        nValue = 0.35;

    return nValue;
}

function float ValidateFriendlyFire(string AValue)
{
    local float nValue;

    // If AValue is not a valid float, the result of this cast will be zero.
    nValue = float(AValue);

    if((nValue < 0) || (nValue > 1))
        nValue = 0;

    return nValue;
}

/*
function int ValidateFragLimit(string AValue)
{
    local int nValue;

    // If AValue is not a valid int, the result of this cast will be zero.
    nValue = int(AValue);

    if((nValue <= 0) || (nValue >= 1000))
        nValue = 30;

    return nValue;
}
*/

function int ValidateBotSkill(string AValue)
{
    local int nValue;

    // If AValue is not a valid int, the result of this cast will be zero.
    nValue = int(AValue);

    if((nValue < 0) || (nValue > 7))
        nValue = 2;

    return nValue;
}

function int ValidateMinPlayers(string AValue)
{
    local int nValue;

    // If AValue is not a valid int, the result of this cast will be zero.
    nValue = int(AValue);

    if((nValue < 0) || (nValue > 32))
        nValue = 2;

    return nValue;
}


function int ValidateMaxTeamScore(string AValue)
{
    local int nValue;

    // If AValue is not a valid int, the result of this cast will be zero.
    nValue = int(AValue);

    if((nValue <= 0) || (nValue >= 1000))
    {
        if(mArena == ARENA_ctf)
            nValue = 5;
        else if (isDeathMatch())
            nValue = 30;
        else
            nValue = 100;
    }

    return nValue;
}

function int ValidateTimeLimit(string AValue)
{
    local int nValue;

    // If AValue is not a valid int, the result of this cast will be zero.
    nValue = int(AValue);

    if((nValue < 0) || (nValue >= 1000))
        nValue = 20;

    return nValue;
}

function int ValidateGravity(string AValue)
{
    local int nValue;

    // If AValue is not a valid int, the result of this cast will be zero.
    nValue = int(AValue);

    // 1000 is normal gravity.
    if((nValue < 1) || (nValue > 1000))
        nValue = 1000;

    return nValue;
}

// Returning False from CheckReplacement() means the actor should self-destruct in their PreBeginPlay().
// Set bSuperRelevant to "0" if i handle it?
function bool CheckReplacement(Actor Other, out byte bSuperRelevant)
{
    //doLog(LOG_error, "CheckReplacement()");

    bSuperRelevant = 1;

    // For our purposes, both 0 and 1000 mean normal gravity.
    if ((cGravity != 0) && (cGravity != 1000))
    {
        if ((Level.Game.IsA('DeathMatchPlus')) && (DeathMatchPlus(Level.Game).bJumpMatch) && (Other.IsA('UT_JumpBoots')))
            return false;

        if (Other.IsA('ZoneInfo'))
        {
            bSuperRelevant = 0;
        	ZoneInfo(Other).ZoneGravity.Z = (cGravity)*(-1);
            doLog(LOG_info, "CheckReplacement(), altering the gravity to "$string(cGravity));
        }
    }

	return true;
}
