class VoiceSpam extends UTServerAdminSpectator config(VoiceSpam);

// ClientVoiceMessage() only receives messages that have the broadcasttype set to 'GLOBAL'. In non team games, this includes all messages (since all taunts are received by all other players).
// In team games, however, only Taunts are sent to all players. This means that acknowledgements, friendly fire, orders, ... are not processed by ClientVoiceMessage() and therefor can't be counted.
// To solve this a second class, UTServerAdminSpectator2, is spawned and made a member of team 1, whilst this class is made a member of team 0.
// UTServerAdminSpectator is normally a member of 255 (none). I don't expect it to be a problem to assign it to a team, since it has bIsPlayer=False and the Spectator flag set, so it shouldn't interfere with scoreboards etc. [although it IS listed in the voicemessage menu..]

var int NumMsg[63]; // Contains the amount of voice messages sent by each player in the game (corresponding to their PlayerID) -- NOTE: max limit is 63, otherwise compiler quits with error: "Variable is too large (256 bytes, 255 max)" when trying to access this array from S2
var string BlockIP[63]; // Contains the IP addresses of players that are blocked from sending voice messages (if bReinforce=True)
var config int MaxMsg; // After how many messages should the player be blocked from sending messages?
var config bool bReinforce; // Whether we should reinforce the voice block if the offending player rejoins the current match
var config bool bPlayerMsg; // Wheter we should inform the player when the voice block takes effect
var config string MsgToPlayer; // Message to send if bPlayerMsg=True
var config bool bBroadcast; // Whether we should tell other players this player is now muted (NOT if bDisableTaunts is active -it would defy the purpose- AND NOT on rejoins)
var config string MsgToBroadcast; // Message to broadcast to other players if bBroadcast=True
var config bool bTeamGames; // True if we should also count team messages
var config bool bDisableTaunts; // True to disable taunts altogether (we should set the VoiceType to none for every player in the game)

function PostBeginPlay()
{
	local pawn p;
	local UTServerAdminSpectator2 S2; // Second MessagingSpectator required to capture team messages

	// SaveConfig(); also saves the properties of PlayerPawn and Pawn since we extend UTServerAdminSpectator, so let's not ;)
	
	// Mark ourselves as a spectator and non-player
	Self.PlayerReplicationInfo.bIsSpectator=true;
	Self.bIsPlayer=false;
	
	if(bBroadcast) if(InStr(MsgToBroadcast,"%Player%") == -1) MsgToBroadcast="The voice message capabilities of %Player% have been revoked.";  // The user has specified an invalid message to broadcast: it doesn't contain the %Player% string. So use default instead.

	if(bReinforce || bDisableTaunts) SetTimer(2.0,true); // If bReinforce or bDisableTaunts, all players are checked every 2 seconds to see if we should reinforce their block
	if(Level.Game.bTeamGame && bTeamGames && !bDisableTaunts){ 
		S2 = Spawn(class 'UTServerAdminSpectator2'); // Spawn S2 to listen for messages from the blue team
		Self.PlayerReplicationInfo.Team=0; // Listen for messages from players from the RED team [note: only ONE actor should handle non-team messages!!! or they get counted double]
	}
}

function Timer() // Note: could also be done in ModifyPlayer(), but then we'd have to worry about the multiple calls made at match start. A timer is more straightforward in this case.
{
	local int i;
	local pawn p;
	
	for (p=Level.PawnList; p!=None; p=p.NextPawn)
	if(p.IsA('PlayerPawn') && !p.IsA('Spectator')) // No need to check anything but players
	if(bDisableTaunts){ if(p.PlayerReplicationInfo.VoiceType!=none){ p.PlayerReplicationInfo.VoiceType=none; // Taunts are disabled in whole
			if(bPlayerMsg) DoMsg(p); } // Send message
	}else{
		for(i=0; i<64 && BlockIP[i]!=""; i++)
		if(p.PlayerReplicationInfo.VoiceType!=none && BlockIP[i]==IP(PlayerPawn(p))){ // If IP address is on the block list and player can still send messages...
			p.PlayerReplicationInfo.VoiceType=none; // Reinforce voice block
			if(bPlayerMsg) DoMsg(p);
			// Don't broadcast revoked message on rejoins
		}
	}
}

function ClientVoiceMessage(PlayerReplicationInfo Sender, PlayerReplicationInfo Recipient, name messagetype, byte messageID)
{
	local pawn p;
	local int i;
	
	if(string(messagetype)~="AUTOTAUNT") return; // Don't count AutoTaunts. S2 will also ignore normal Taunts, since they are handled by S1 (Self).
	
	for (p=Level.PawnList; p!=None; p=p.NextPawn)
	if((p.IsA('PlayerPawn') && !p.IsA('Spectator') || p.IsA('Bot')) && p.PlayerReplicationInfo.PlayerID==Sender.PlayerID) break; // p is now at our Sender
	
	if(p.IsA('Bot')) return; // Ignore bots

	NumMsg[Sender.PlayerID]++; // Add +1 to the voice messages count of this player

	if(NumMsg[Sender.PlayerID]==MaxMsg){ // The number of voice messages sent has reached MaxMsg (we use equal (==) since this message will still be sent; the block takes effect after this msg)
		p.PlayerReplicationInfo.VoiceType=none; // Set VoiceType to none so that PlayerPawn.ClientVoiceMessage() returns prematurely when player wants to send a message
		if(bPlayerMsg) DoMsg(p);
		if(bBroadcast) DoMsg(p,true); // Tell other players this player is now muted
		if(!bReinforce) return; // We are done here..
		for(i=0; i<64; i++)
		if(BlockIP[i]==""){ BlockIP[i]=IP(PlayerPawn(p)); break; } // Find an empty spot on the BlockIP[x] list and add our IP address
	}
}

function DoMsg(pawn badPlayer, optional bool bB)
{
	local pawn p;
	local string tmpMsg;

	if(bB){ // This is a broadcast call
		tmpMsg=MsgToBroadcast; // We use a temporary variable so our dynamic MsgToBroadcast isn't replaced with a version containing a static playername
		ReplaceText(tmpMsg, "%Player%", badPlayer.PlayerReplicationInfo.Playername); // Construct the MsgToBroadcast by substituting %Player% with the Playername
		for (p=Level.PawnList; p!=None; p=p.NextPawn)
		if(p.IsA('PlayerPawn') && p!=badPlayer) p.ClientMessage(tmpMsg); // Send to message to all other players (NOT to the muted player).
	}else{
		badPlayer.ClientMessage(MsgToPlayer);} // Send the message to the muted player
}

function string IP(PlayerPawn p)
{
	local int i;
	local string IP;

	IP = p.GetPlayerNetworkAddress();
	i = InStr(IP, ":");
	if (i != -1)
	IP = Left(IP, i);
	return IP;
}

defaultproperties
{
	MaxMsg=20
	bReinforce=True
	bPlayerMsg=True
	MsgToPlayer="You have reached the maximum amount of taunts allowed for this match."
	bBroadcast=True
	MsgToBroadcast="The voice message capabilities of %Player% have been revoked."
	bTeamGames=True
	bDisableTaunts=False
}