class SmartUTScoreBoard extends UnrealCTFScoreBoard abstract;

#exec texture IMPORT NAME=faceless File=Textures\faceless.pcx GROUP=SmartUT

var ScoreBoard NormalScoreBoard;
var class<ScoreBoard> NormalScoreBoardClass;
var SmartUTGameReplicationInfo SmartUTGame;
var SmartUTPRI OwnerStats;

var int TryCount;
var PlayerPawn PlayerOwner;

var string ScoreText, PointsText, SepText, MoreText, HeaderText;
var int LastSortTime;
var int MaxMeterWidth;
var byte ColorChangeSpeed, RowColState;
var Color White, Gray, DarkGray, Yellow, RedTeamColor, BlueTeamColor, RedHeaderColor, BlueHeaderColor, StatsColor, FooterColor, HeaderColor, TinyInfoColor, HeaderTinyInfoColor;
var float StatsTextWidth, StatHeight, MeterHeight, NameHeight, ColumnHeight, StatBlockHeight;
var float RedStartX, BlueStartX, ColumnWidth, StatWidth, StatsHorSpacing, ShadingSpacingX, HeaderShadingSpacingY, ColumnShadingSpacingY;
var float StartY, StatLineHeight, StatBlockSpacing, StatIndent;
var TournamentGameReplicationInfo pTGRI;
var PlayerReplicationInfo pPRI;
var Font StatFont, CapFont, FooterFont, GameEndedFont, PlayerNameFont, FragsFont, TinyInfoFont;
var Font PtsFont22, PtsFont20, PtsFont18, PtsFont16, PtsFont14, PtsFont12;

var int MaxFrags, MaxDeaths;
var int TotShieldBelts, TotAmps;

// CountryFlags
struct FlagData {
	var string Prefix;
	var texture Tex;
};
var FlagData FD[32]; // there can be max 32 so max 32 different flags
var int saveindex; // new loaded flags will be saved in FD[index]

function PostBeginPlay()
{
  super.PostBeginPlay();

  PlayerOwner = PlayerPawn( Owner );
  pTGRI = TournamentGameReplicationInfo( PlayerOwner.GameReplicationInfo );
  pPRI = PlayerOwner.PlayerReplicationInfo;
  LastSortTime = -100;

  // Preload
  PtsFont22 = Font( DynamicLoadObject( "LadderFonts.UTLadder22", class'Font' ) );
  PtsFont20 = Font( DynamicLoadObject( "LadderFonts.UTLadder20", class'Font' ) );
  PtsFont18 = Font( DynamicLoadObject( "LadderFonts.UTLadder18", class'Font' ) );
  PtsFont16 = Font( DynamicLoadObject( "LadderFonts.UTLadder16", class'Font' ) );
  PtsFont14 = Font( DynamicLoadObject( "LadderFonts.UTLadder14", class'Font' ) );
  PtsFont12 = Font( DynamicLoadObject( "LadderFonts.UTLadder12", class'Font' ) );

  SpawnNormalScoreBoard();
  if( NormalScoreBoard == None )
	SetTimer( 1.0 , True );
}

function SpawnNormalScoreBoard()
{
  if( SmartUTGame == None )
  {
    ForEach AllActors( class'SmartUTGameReplicationInfo', SmartUTGame )
		break;
  }
 
  if( SmartUTGame != None && SmartUTGame.NormalScoreBoardClass == None )
  {
    Log( "Unable to identify original ScoreBoard type. Retrying in 1 second." , 'SmartUT' );
    return;
  }

  if( SmartUTGame != None && SmartUTGame.NormalScoreBoardClass == self.Class )
  {
    NormalScoreBoard = Spawn(NormalScoreBoardClass, PlayerOwner );
    Log( "Cannot use itself. Using the default ScoreBoard instead." , 'SmartUT' );
    return;
  }

  if( SmartUTGame != None && SmartUTGame.NormalScoreBoardClass != None )
  {
    NormalScoreBoard = Spawn( SmartUTGame.NormalScoreBoardClass, PlayerOwner );
    Log( "Determined and spawned original scoreboard as" @ NormalScoreBoard, 'SmartUT' );
  }
}

// In the case of the 'normal scoreboard' not being replicated properly, try every second to see if it has.
function Timer()
{
  if( NormalScoreBoard == None )
  {
    TryCount++;
    SpawnNormalScoreBoard();
  }

  if( NormalScoreBoard != None )
  {
	SetTimer(0.0, false);
  }
  else if( TryCount > 3 )
  {
    Log( "Given up. Using the default ScoreBoard instead." , 'SmartUT' );

    if( NormalScoreBoard == None )
    {
      NormalScoreBoard = Spawn( NormalScoreBoardClass, PlayerOwner );
      Log( "Spawned as" @ NormalScoreBoard, 'SmartUT' );
    }
	SetTimer( 0.0, False );    
  }
}

function ShowScores( Canvas C ) {} // Override

function SmartUTShowScores(Canvas C) {} // Override
 
function ShowIndicator( Canvas C )
{
  local float BlockLen, LineHeight;

  if (C==None) return;
  if (C==None || OwnerStats==None) return;
  
  C.DrawColor.R = OwnerStats.IndicatorVisibility;
  C.DrawColor.G = OwnerStats.IndicatorVisibility;
  C.DrawColor.B = OwnerStats.IndicatorVisibility;
  C.Style = ERenderStyle.STY_Translucent;
  C.Font = C.SmallFont;
  C.StrLen( "Scoreboard:", BlockLen, LineHeight );
  C.SetPos( C.ClipX - BlockLen - 16, 16 );
  C.DrawText( "Scoreboard:" );
  C.SetPos( C.ClipX - BlockLen, 16 + LineHeight );
  C.DrawText( "Default" );
  C.SetPos( C.ClipX - BlockLen, 16 + 2 * LineHeight );
  C.DrawText( "SmartUT" );
  if( OwnerStats.bViewingStats )
	C.SetPos( C.ClipX - BlockLen - 16, 16 + 2 * LineHeight );
  else
	C.SetPos( C.ClipX - BlockLen - 16, 16 + LineHeight );
  C.DrawIcon( texture'UWindow.MenuTick', 1 );
  C.Style = ERenderStyle.STY_Normal;

  if( Level.TimeSeconds - OwnerStats.IndicatorStartShow > 2 )
	OwnerStats.IndicatorVisibility = 0;
}

function InitStatBoardConstPos( Canvas C )
{
  local float Nil, LeftSpacingPercent, MidSpacingPercent, RightSpacingPercent;

  CapFont = Font'LEDFont2'; //Font( DynamicLoadObject( "UWindowFonts.UTFont40", class'Font' ) );
  FooterFont = MyFonts.GetSmallestFont( C.ClipX );
  GameEndedFont = MyFonts.GetHugeFont( C.ClipX );
  PlayerNameFont = MyFonts.GetBigFont( C.ClipX );
  TinyInfoFont = C.SmallFont;

  if( PlayerNameFont == PtsFont22 ) FragsFont = PtsFont18;
  else if( PlayerNameFont == PtsFont20 ) FragsFont = PtsFont18;
  else if( PlayerNameFont == PtsFont18 ) FragsFont = PtsFont14;
  else if( PlayerNameFont == PtsFont16 ) FragsFont = PtsFont12;
  else FragsFont = font'SmallFont';

  C.Font = PlayerNameFont;
  C.StrLen( "Player", Nil, NameHeight );

  StartY = ( 120.0 / 1024.0 ) * C.ClipY;
  ColorChangeSpeed = 100; // Influences how 'fast' the color changes from white to green. Higher = faster.

  LeftSpacingPercent = 0.075;
  MidSpacingPercent = 0.15;
  RightSpacingPercent = 0.075;
  RedStartX = LeftSpacingPercent * C.ClipX;
  ColumnWidth = ( ( 1 - LeftSpacingPercent - MidSpacingPercent - RightSpacingPercent ) / 2 * C.ClipX );
  BlueStartX = RedStartX + ColumnWidth + ( MidSpacingPercent * C.ClipX );
  ShadingSpacingX = ( 10.0 / 1024.0 ) * C.ClipX;
  HeaderShadingSpacingY = ( 32 - NameHeight ) / 2 + ( ( 4.0 / 1024.0 ) * C.ClipX );
  ColumnShadingSpacingY = ( 10.0 / 1024.0 ) * C.ClipX;

  StatsHorSpacing = ( 5.0 / 1024.0 ) * C.ClipX;
  StatIndent = ( 32 + StatsHorSpacing ); // For face + flag icons

  InitStatBoardDynamicPos( C );
}

function InitStatBoardDynamicPos( Canvas C , optional int Rows , optional int Cols , optional Font NewStatFont , optional float LineSpacing , optional float BlockSpacing )
{
  if( Rows == 0 ) Rows = 3;
  if( Cols == 0 ) Cols = 2;
  if( LineSpacing == 0 ) LineSpacing = 0.9;
  if( BlockSpacing == 0 ) BlockSpacing = 1;

  if( Rows == 2 && Cols == 3 ) RowColState = 1;
  else RowColState = 0;

  StatWidth = ( ( ColumnWidth - StatIndent ) / Cols ) - ( StatsHorSpacing * ( Cols - 1 ) );

  if( NewStatFont == None ) StatFont = MyFonts.GetSmallestFont( C.ClipX );
  else StatFont = NewStatFont;
  C.Font = StatFont;
  C.StrLen( "Suicides: 999", StatsTextWidth, StatHeight );

  MaxMeterWidth = StatWidth - StatsTextWidth - StatsHorSpacing;
  StatLineHeight = StatHeight * LineSpacing;
  MeterHeight = Max( 1, StatLineHeight * 0.3 );
  StatBlockSpacing = StatLineHeight * BlockSpacing;

  StatBlockHeight = Rows * StatLineHeight;

	if (pTGRI.Teams[0].Size > pTGRI.Teams[1].Size )
		ColumnHeight = pTGRI.Teams[0].Size * ( NameHeight + StatBlockHeight + StatBlockSpacing ) - StatBlockSpacing;
	else
		ColumnHeight = pTGRI.Teams[1].Size * ( NameHeight + StatBlockHeight + StatBlockSpacing ) - StatBlockSpacing;
}

function CompressStatBoard( Canvas C , optional int Level )
{
  local float EndY, Nil, DummyY;

  C.Font = FooterFont;
  C.StrLen( "Test", Nil, DummyY );

  EndY = StartY + ColumnHeight + ( ColumnShadingSpacingY * 2 ) + NameHeight + HeaderShadingSpacingY;
  if( EndY > C.ClipY - DummyY * 5 )
  {
    if( Level == 0 )
    {
      InitStatBoardDynamicPos( C, , , , 0.8 );
    }
    else if( Level == 1 )
    {
      InitStatBoardDynamicPos( C, 2, 3 );
    }
    else if( Level == 2 )
    {
      InitStatBoardDynamicPos( C, 2, 3, Font( DynamicLoadObject( "UWindowFonts.Tahoma10", class'Font' ) ) , 1.0 , 1.0 );
    }
    else
    {
      // We did all the compression we can do. Draw 'More' labels later.
      // First find the columnheight for the amount of players that fit on it.
      ColumnHeight = int( ( C.ClipY - ( EndY - ColumnHeight ) - DummyY * 5 + StatBlockSpacing ) / ( NameHeight + StatBlockHeight + StatBlockSpacing ) )
        * ( NameHeight + StatBlockHeight + StatBlockSpacing ) - StatBlockSpacing;
      return;
    }
    // Did some compression, see if we need more.
    CompressStatBoard( C , Level + 1 );
  }
  // No compression at all or no more compression needed.
  return;
}

/*
 * Draw a specific stat
 * X, Y = Upper left corner of stats ( row,col: 1,1)
 */
function DrawStatType( Canvas C, int X, int Y, int Row, int Col, string Label, int Count, int Total )
{
  local float Size, DummyY;
  local int ColorChange, M;

  X += StatIndent + ( ( StatWidth + StatsHorSpacing ) * ( Col - 1 ) );
  Y += ( StatLineHeight * ( Row - 1 ) );

  C.DrawColor = StatsColor;
  C.SetPos( X, Y );
  C.DrawText( Label );
  C.StrLen( Count, Size, DummyY );
  C.SetPos( X + StatsTextWidth - Size, Y );
  C.DrawText( Count ); //text
  
  if (Total>0)
  {
	  if( Count > 0 )
	  {
	    ColorChange = ColorChangeSpeed * loge( Count );
	    if( ColorChange > 255 ) ColorChange = 255;
	    C.DrawColor.R = StatsColor.R - ColorChange;
	    C.DrawColor.B = StatsColor.B - ColorChange;
	  }
	  M = GetMeterLength( Count, Total );
	  C.SetPos( X + StatsTextWidth + StatsHorSpacing, Y + ( ( StatHeight - MeterHeight ) / 2 ) );
	  C.DrawRect( texture'meter', M, MeterHeight ); //meter
	}
}

function DrawStatTypeText( Canvas C, int X, int Y, int Row, int Col, string Label, int Count, string Name)
{
  local float Size, DummyY;
  local int ColorChange, M;

  X += StatIndent + ( ( StatWidth + StatsHorSpacing ) * ( Col - 1 ) );
  Y += ( StatLineHeight * ( Row - 1 ) );

  C.DrawColor = StatsColor;
  C.SetPos( X, Y );
  C.DrawText( Label );
  C.StrLen( Count, Size, DummyY );
  C.SetPos( X + StatsTextWidth - Size, Y );
  C.DrawText( Count );
  C.SetPos( X + StatsTextWidth + StatsHorSpacing, Y );
  C.DrawText( Name );
    
}

function DrawFooters( Canvas C )
{
  local float DummyX, DummyY, Nil, X1, Y1;
  local string TextStr;
  local string TimeStr;
  local int Hours, Minutes, Seconds;

  //if (C==None) return;
  
  C.bCenter = True;
  C.Font = FooterFont;

  // Display server info in bottom center
  C.DrawColor = FooterColor;
  C.StrLen( "Test", DummyX, DummyY );
  C.SetPos( 0, C.ClipY - DummyY );
  TextStr = "Playing" @ Level.Title @ "on" @ pTGRI.ServerName;
  if( SmartUTGame.TickRate > 0 ) TextStr = TextStr @ "(TR:" @ SmartUTGame.TickRate $ ")";
  C.DrawText( TextStr );

  // Draw Time
  if( bTimeDown || ( PlayerOwner.GameReplicationInfo.RemainingTime > 0 ) )
  {
    bTimeDown = True;
    if( PlayerOwner.GameReplicationInfo.RemainingTime <= 0 )
    {
      TimeStr = RemainingTime $ "00:00";
    }
    else
    {
      Minutes = PlayerOwner.GameReplicationInfo.RemainingTime / 60;
      Seconds = PlayerOwner.GameReplicationInfo.RemainingTime % 60;
      TimeStr = RemainingTime $ TwoDigitString( Minutes ) $ ":" $ TwoDigitString( Seconds );
    }
  }
  else
  {
    Seconds = PlayerOwner.GameReplicationInfo.ElapsedTime;
    Minutes = Seconds / 60;
    Hours = Minutes / 60;
    Seconds = Seconds - ( Minutes * 60 );
    Minutes = Minutes - ( Hours * 60 );
    TimeStr = ElapsedTime $ TwoDigitString( Hours ) $ ":" $ TwoDigitString( Minutes ) $ ":" $ TwoDigitString( Seconds );
  }
  C.SetPos( 0, C.ClipY - 2 * DummyY );
  C.DrawText( "Current Time:" @ GetTimeStr() @ "|" @ TimeStr );

  // Draw Author
  C.StrLen( HeaderText, DummyX, Nil );
  C.Style = ERenderStyle.STY_Normal;
  C.DrawColor = Yellow;
  C.SetPos( 0, C.ClipY - 4 * DummyY );
  C.DrawText( "["@HeaderText@" *beta | (W)2007 iDeFiX ]");
  C.bCenter = False;
}

function DrawHeader( Canvas C )
{
  local float DummyX, DummyY;

  if( pTGRI.GameEndedComments == "" ) return;

  C.Font = GameEndedFont;
  C.StrLen( pTGRI.GameEndedComments, DummyX, DummyY );

  C.DrawColor = DarkGray;
  C.Style = ERenderStyle.STY_Translucent;
  C.SetPos( C.ClipX / 2 - DummyX / 2 + 2, DummyY + 2 );
  C.DrawText( pTGRI.GameEndedComments );

  C.DrawColor = HeaderColor;
  C.Style = ERenderStyle.STY_Normal;
  C.SetPos( C.ClipX / 2 - DummyX / 2, DummyY );
  C.DrawText( pTGRI.GameEndedComments );
}

/*
 * Returns time and date in a string.
 */
function string GetTimeStr()
{
  local string Mon, Day, Min;

  Min = string( PlayerOwner.Level.Minute );
  if( int( Min ) < 10 ) Min = "0" $ Min;

  switch( PlayerOwner.Level.month )
  {
    case  1: Mon = "Jan"; break;
    case  2: Mon = "Feb"; break;
    case  3: Mon = "Mar"; break;
    case  4: Mon = "Apr"; break;
    case  5: Mon = "May"; break;
    case  6: Mon = "Jun"; break;
    case  7: Mon = "Jul"; break;
    case  8: Mon = "Aug"; break;
    case  9: Mon = "Sep"; break;
    case 10: Mon = "Oct"; break;
    case 11: Mon = "Nov"; break;
    case 12: Mon = "Dec"; break;
  }

  switch( PlayerOwner.Level.dayOfWeek )
  {
    case 0: Day = "Sunday";    break;
    case 1: Day = "Monday";    break;
    case 2: Day = "Tuesday";   break;
    case 3: Day = "Wednesday"; break;
    case 4: Day = "Thursday";  break;
    case 5: Day = "Friday";    break;
    case 6: Day = "Saturday";  break;
  }

  return Day @ PlayerOwner.Level.Day @ Mon @ PlayerOwner.Level.Year $ "," @ PlayerOwner.Level.Hour $ ":" $ Min;
}

/*
 * Sort PlayerReplicationInfo's on score.
 */
function SortScores( int N )
{
  local byte i, j;
  local bool bSorted;
  local SmartUTPRI PlayerStats1, PlayerStats2;

  // Copy PRI array except for spectators.
  j = 0;
  for( i = 0; i < N; i++ )
  {
    if( pTGRI.priArray[i] == None ) break;
    if( pTGRI.priArray[i].bIsSpectator && !pTGRI.priArray[i].bWaitingPlayer ) continue;
    Ordered[j] = pTGRI.priArray[i];
    j++;
  }
  // Clear the remaining entries.
  for( i = j; i < N; i++ )
  {
    Ordered[i] = None;
  }

  for( i = 0; i < N; i++)
  {
    bSorted = True;
    for( j = 0; j < N - 1; j++)
    {
      if( Ordered[j] == None || Ordered[j+1] == None ) break;

      if( Ordered[j].Score < Ordered[j+1].Score )
      {
        SwapOrdered( j, j + 1 );
        bSorted = False;
      }
      else if( Ordered[j].Score == Ordered[j+1].Score )
      {
        PlayerStats1 = SmartUTGame.GetStatsByPRI( Ordered[j] );
        PlayerStats2 = SmartUTGame.GetStatsByPRI( Ordered[j+1] );
        if( PlayerStats1 != None && PlayerStats2 != None )
        {
          if( PlayerStats1.Score < PlayerStats2.Score )
          {
            SwapOrdered( j, j + 1 );
            bSorted = False;
          }
          else if( PlayerStats1.Score == PlayerStats2.Score )
          {
            if( Ordered[j].Deaths > Ordered[j+1].Deaths )
            {
              SwapOrdered( j, j + 1 );
              bSorted = False;
            }
          }
        }
      }
    }
    if( bSorted ) break;
  }
}

function int GetMeterLength( int A, int B )
{
  local int Result;

  if( B == 0 ) return 0;
  Result = ( A * MaxMeterWidth ) / B;

  if( Result > MaxMeterWidth ) return MaxMeterWidth;
  else return Result;
}

/*
 * Used for sorting.
 */
function SwapOrdered( byte A, byte B )
{
  local PlayerReplicationInfo Temp;
  Temp = Ordered[A];
  Ordered[A] = Ordered[B];
  Ordered[B] = Temp;
}

/*
 * Recalculate the totals for displaying meters on the scoreboards.
 * This way it doesn't get calculated every tick.
 */
function RecountNumbers()
{
  local byte ID, i;
  local SmartUTPRI PlayerStats;

  MaxFrags = 0;
  TotShieldBelts = 0;
  TotAmps = 0;

  for( i = 0; i < 32; i++ )
  {
    if( Ordered[i] == None )
		break;
    if( Ordered[i].bIsSpectator && !Ordered[i].bWaitingPlayer )
		continue;

    ID = Ordered[i].PlayerID;

    PlayerStats = SmartUTGame.GetStatsByPRI( Ordered[i] );
    if( PlayerStats != None )
    {
      if( Ordered[i].Score > MaxFrags )
		MaxFrags = Ordered[i].Score;
      TotShieldBelts += PlayerStats.ShieldBelts;
      TotAmps += PlayerStats.Amplifiers;
    }
  }
}

// CountryFlags
function int GetFlagIndex(string Prefix)
{
	local int i;
	for(i=0;i<32;i++)
		if(FD[i].Prefix == Prefix)
			return i;
	FD[saveindex].Prefix=Prefix;
	FD[saveindex].Tex=texture(DynamicLoadObject(SmartUTGame.CountryFlagsPackage$"."$Prefix, class'Texture'));
	i=saveindex;
	saveindex = (saveindex+1) % 256;
	return i;
}

defaultproperties
{
    NormalScoreBoardClass=Class'Engine.ScoreBoard'
    ScoreText="Score"
    PointsText="Pts"
    SepText=" / "
    MoreText="More..."
    HeaderText="SmartUT"
    White=(R=255,G=255,B=255,A=0),
    Gray=(R=128,G=128,B=128,A=0),
    DarkGray=(R=32,G=32,B=32,A=0),
    Yellow=(R=255,G=255,B=0,A=0),
    RedTeamColor=(R=255,G=0,B=0,A=0),
    BlueTeamColor=(R=0,G=128,B=255,A=0),
    RedHeaderColor=(R=64,G=0,B=0,A=0),
    BlueHeaderColor=(R=0,G=32,B=64,A=0),
    StatsColor=(R=255,G=255,B=255,A=0),
    FooterColor=(R=255,G=255,B=255,A=0),
    HeaderColor=(R=255,G=255,B=0,A=0),
    TinyInfoColor=(R=128,G=128,B=128,A=0),
    HeaderTinyInfoColor=(R=192,G=192,B=192,A=0),
}
