class SmartUT expands Mutator config( SmartUT ) abstract;

#exec texture IMPORT NAME=meter FILE=Textures\meter.pcx GROUP=SmartUT
#exec texture IMPORT NAME=shade File=Textures\shade.pcx GROUP=SmartUT

/* Server Vars */
var SmartUTGameReplicationInfo SmartUTGame;
var string Version;
var string GameTieMessage;
var bool bWarmup, bForcedEndGame, bTournamentGameStarted, bCheckReady;
var TeamInfo LastLeadTeam;
var CTFFlag ReadyFlag;

// Override
var class<SmartUTEndStats>  SmartUTEndStatsClass;
var class<SmartUTGameReplicationInfo> SmartUTGameReplicationInfoClass;
var class<SmartUTScoreBoard>  SmartUTScoreBoardClass;
var class<SmartUTServerInfo> SmartUTServerInfoClass;

/* Client Vars */
var bool bClientJoinPlayer, bGameEnded;
var int LogoCounter, DrawLogo;
var PlayerPawn PlayerOwner;
var FontInfo MyFonts;
var TournamentGameReplicationInfo pTGRI;
var PlayerReplicationInfo pPRI;
var ChallengeHUD MyHUD;
var Color RedTeamColor, BlueTeamColor, GreenTeamColor, GoldTeamColor, White, Gray;

/* Server Vars Configurable */
var() config bool bEnabled;
var() config bool bShowBonusConsoleMsg;
var() config bool bEnhancedMultiKill, bAfterGodLikeMsg;
var() config byte EnhancedMultiKillBroadcast;
var() config bool bServerInfo;
var() config bool bShowExtraInfo;
var() config bool bNewSpawnKillScoring, bSpawnKillDetection, bShowSpawnKillerGlobalMsg;
var() config float SpawnKillTime;

var() config bool bStatsDrawFaces;
var() config string CountryFlagsPackage;
var() config bool bReadyFlag;

var() config int SpreeLevelBonus, MultiKillLevelBonus;
var() config int KillBonus, SuicidePenalty, SpawnKillPenalty;
var() config bool bShowLongRangeMsg;
var() config bool bPlayLeadSound, bPlay30SecSound;
var() config bool bEnableOvertimeControl, bOvertime, bRememberOvertimeSetting;

/*
 * Check if we should spawn a SmartUT instance.
 * This check doesn't seem to work properly in PostBeginPlay, hence here.
 */
event Spawned()
{
	super.Spawned();

	SmartUTGame = Spawn(SmartUTGameReplicationInfoClass);
	if (!ValidateSmartUTMutator())
	{
		SmartUTGame.Destroy();
		Destroy();
	}
}

/*
 * Get the original Scoreboard and store for SmartTDMScoreBoard reference.
 */
function PreBeginPlay()
{
	local Mutator M;

	super.PreBeginPlay();

	SmartUTGame.NormalScoreBoardClass = Level.Game.ScoreBoardType;
	Level.Game.ScoreBoardType = SmartUTScoreBoardClass;
	Log( "Info: Original Scoreboard determined as" @ SmartUTGame.NormalScoreBoardClass, 'SmartUT' );

	// Change F2 Server Info screen, compatible with UTPure
	if (bServerInfo)
	{
		class<ChallengeHUD>( Level.Game.HUDType ).default.ServerInfoClass = SmartUTServerInfoClass;
		for (M = Level.Game.BaseMutator; M!=None; M=M.NextMutator)
		{
			if( M.IsA( 'UTPure' ) ) // Let UTPure rehandle the scoreboard
			{
				M.PreBeginPlay();
				SmartUTGame.bServerInfoSetServerSide = True; // No need for the old fashioned way - it can be set server side.
				Log( "Info: Notified UTPure HUD to use a SmartUT ServerInfo.", 'SmartUT' );
				break;
			}
		}	
	    if (SmartUTGame.bServerInfoSetServerSide && Level.Game.HUDType.Name != 'PureCTFHUD')
	    {
	      // In this scenario another mod intervered and we still have to do it the old fashion way.
	      SmartUTGame.bServerInfoSetServerSide = False;
	      Log( "Info: HUD is not the UTPure HUD but" @ Level.Game.HUDType.Name $ ", so SmartUT ServerInfo will be set clientside.", 'SmartUT' );
	    }
		if (!SmartUTGame.bServerInfoSetServerSide)
			SmartUTGame.DefaultHUDType = Level.Game.HUDType; // And in the old fashion way, the client will have to know the current HUD type.
	}
	else
	{
		SmartUTGame.bServerInfoSetServerSide = True; // We didn't change anything, but neither do we want clientside intervention.
	}
	
	bCheckReady = false;
	if (bReadyFlag && DeathMatchPlus(Level.Game).bTournament)
	{
		ReadyFlag = Spawn(class'SmartUTFlag');
		bCheckReady = (ReadyFlag!=None);
		if (!bCheckReady)
			log("Error: bReadyFlag enabled, but unable to spawn ReadyFlag", 'SmartUT');
	}
	
}
/*
 * Startup and initialize.
 */
function PostBeginPlay()
{
	local Mutator m;
	
	Level.Game.Spawn( class'SmartUTSpawnNotify' );

	SaveConfig(); // Create the .ini if its not already there.

	Level.Game.RegisterMessageMutator( self );

	// Since we have problem replicating config variables...
	SmartUTGame.bStatsDrawFaces = bStatsDrawFaces;
	SmartUTGame.bPlay30SecSound = bPlay30SecSound;
	SmartUTGame.bStatsDrawFaces = bStatsDrawFaces;
	SmartUTGame.bShowExtraInfo = bShowExtraInfo;
	
	if (!bRememberOvertimeSetting)
		bOvertime = True;
	
	// Works serverside!
	if (bEnhancedMultiKill)
		Level.Game.DeathMessageClass = class'SmartUTEnhancedDeathMessagePlus';
		
	SmartUTGame.EndStats = Spawn( SmartUTEndStatsClass, self);
	SmartUTGame.CountryFlagsPackage = CountryFlagsPackage;
	
	super.PostBeginPlay();
	
	if (Level.NetMode == NM_DedicatedServer || Role == ROLE_Authority)
		SetTimer(1.0, True);
	
	// Detect Tournament mode (official games)
	bWarmup = false;
	if (TeamGamePlus(Level.Game).bTournament && TeamGamePlus(Level.Game).CountDown>0)
		bWarmup = true;

	Log("Info:"@self.class.name @ Version @ "loaded successfully.", 'SmartUT' );
}

/*
 * Returns True or False whether to keep this SmartTDM mutator instance, and sets bInitialized accordingly.
 */
function bool ValidateSmartUTMutator()
{
	local Mutator M;
	local bool bRunning;

	M = Level.Game.BaseMutator;
	while( M != None )
	{
		if( M != Self && M.Class == Self.Class )
		{
			bRunning = True;
			break;
		}
		M = M.NextMutator;
	}

	if (!bEnabled)
		Log( "Instance" @ Name @ "not loaded because bEnabled in .ini = False.", 'SmartUT' );
	else if (TeamGamePlus(Level.Game)==None)
		Log( "Instance" @ Name @ "not loaded because gamestyle is no teangame.", 'SmartUT' );
	else if (bRunning)
		Log( "Instance" @ Name @ "not loaded because it is already running.", 'SmartUT' );
	else
		SmartUTGame.bInitialized = True;

	return SmartUTGame.bInitialized;
}

function ModifyPlayer( Pawn Other )
{
	local SmartUTPRI OtherStats;
	local byte ID;
	local string SkinName, FaceName;
  
	OtherStats = SmartUTGame.GetStats( Other );
	if (OtherStats==None) return;

	if (!OtherStats.bHadFirstSpawn)
	{
		OtherStats.bHadFirstSpawn = True;

		// Additional logging, useful for player tracking
		if (Level.Game.LocalLog != None && PlayerPawn(Other)!=None && Other.bIsPlayer)
		{
		    ID = PlayerPawn(Other).PlayerReplicationInfo.PlayerID;
		    Level.Game.LocalLog.LogSpecialEvent( "IP", ID, PlayerPawn( Other ).GetPlayerNetworkAddress() );
		    Level.Game.LocalLog.LogSpecialEvent( "player", "NetSpeed", ID, PlayerPawn( Other ).Player.CurrentNetSpeed );
		    Level.Game.LocalLog.LogSpecialEvent( "player", "Fov", ID, PlayerPawn( Other ).FovAngle );
		    Level.Game.LocalLog.LogSpecialEvent( "player", "VoiceType", ID, Other.VoiceType );
			if (Other.IsA('TournamentPlayer'))
			{
				if( Other.Skin == None )
				{
					Other.static.GetMultiSkin( Other, SkinName, FaceName );
				}
				else
				{
					SkinName = string( Other.Skin );
					FaceName = "None";
				}
				Level.Game.LocalLog.LogSpecialEvent( "player", "Skin", ID, SkinName );
				Level.Game.LocalLog.LogSpecialEvent( "player", "Face", ID, FaceName );
			}
		}	  
	}
	OtherStats.SpawnTime = Level.TimeSeconds;
	
	super.ModifyPlayer( Other );
}

function TournamentGameStarted()
{
	// Fix warmup mode bug + Overtime functionality
	ClearStats();
	if (bEnableOvertimeControl)
	{
		if (!bOvertime)
			BroadcastLocalizedMessage( class'SmartUTCoolMsg', 4 );
		else
			BroadcastLocalizedMessage( class'SmartUTCoolMsg', 3 );
	}
}

function bool PreventDeath( Pawn Victim, Pawn Killer, name DamageType, vector HitLocation )
{
	local PlayerReplicationInfo VictimPRI, KillerPRI;
	local bool bPrevent;
	local Pawn p;
	local SmartUTPRI KillerStats, VictimStats;
	local Inventory aAmp;
	local float TimeAwake;
	local ControlPoint cp;

	bPrevent = super.PreventDeath( Victim, Killer, DamageType, HitLocation );
	if (bWarmup || bPrevent || Victim==None || Killer==None)
		return bPrevent; 

	VictimPRI = Victim.PlayerReplicationInfo;	
	if( VictimPRI==None || !Victim.bIsPlayer || ( VictimPRI.bIsSpectator && !VictimPRI.bWaitingPlayer ) )
		return bPrevent;

	VictimStats = SmartUTGame.GetStats(Victim);
	if (VictimStats!=None)
	{
		ProcessSpreeStats(VictimStats);
		ProcessMultiStats(VictimStats);
		VictimStats.SpawnKillSpree = 0;
	}
	
	// Suicide
	if (Killer==None || Killer==Victim)
	{	
		VictimStats.Suicides++;
		
		if (SuicidePenalty!=0)
			VictimStats.Score -= SuicidePenalty;
		
		return bPrevent;
	}
	
	KillerPRI = Killer.PlayerReplicationInfo;
	if ( KillerPRI==None || !Killer.bIsPlayer || ( KillerPRI.bIsSpectator && !KillerPRI.bWaitingPlayer ) )
		return bPrevent;
		
	// Register long range kills
	if (bShowLongRangeMsg && TournamentPlayer(Killer)!=None)
	{
		if (DamageType!='shot' && DamageType!='decapitated' && DamageType!='Gibbed' && DamageType!='RedeemerDeath' && SuperShockRifle(Killer.Weapon)==None && DamageType!='Eradicated')
		{
			if (VSize(Killer.Location-Victim.Location)>1536)
			{
				// Announce
				if (VSize(Killer.Location-Victim.Location)>3072)
					Killer.ReceiveLocalizedMessage( class'SmartUTCoolMsg', 2, KillerPRI, VictimPRI );
				else
					Killer.ReceiveLocalizedMessage( class'SmartUTCoolMsg', 1, KillerPRI, VictimPRI );
		
				// Log
				if( Level.Game.LocalLog != None )
					Level.Game.LocalLog.LogSpecialEvent("longrangekill", KillerPRI.PlayerID, VictimPRI.PlayerID);
			}
		}
	}

	KillerStats = SmartUTGame.GetStats(Killer);
	if (KillerStats!=None)
	{				
		// Register headshots
		if (DamageType=='decapitated')
			KillerStats.HeadShots++;
		
		KillerStats.Kills++;
		if (KillBonus!=0)
			KillerStats.Score += KillBonus;
		
		// Track multikills
		if (Level.TimeSeconds-KillerStats.LastKillTime<3)
		{
			KillerStats.MultiLevel++;
			if (bEnhancedMultiKill && EnhancedMultiKillBroadcast>0)
			{
				if (KillerStats.MultiLevel+1>=EnhancedMultiKillBroadcast)
					Level.Game.BroadcastMessage( KillerPRI.PlayerName @ class'SmartUTEnhancedMultiKillMessage'.static.GetBroadcastString( KillerStats.MultiLevel ) );
			}
		}
		else
		{
			ProcessMultiStats(KillerStats);
		}
		KillerStats.LastKillTime = Level.TimeSeconds;
		
		// After Godlike
		KillerStats.FragSpree++;
		if( bAfterGodLikeMsg && KillerStats!=None && (KillerStats.FragSpree==30 || KillerStats.FragSpree==35) )
		{
			for(p = Level.PawnList; p!=None; p=p.NextPawn )
			{
				if (p.IsA('TournamentPlayer'))
					p.ReceiveLocalizedMessage( class'SmartUTSpreeMsg', KillerStats.FragSpree / 5 - 1, KillerPRI );
			}
		}

		// Spawnkill detection
		if(bSpawnkillDetection && DamageType!='Gibbed')
		{
			TimeAwake = Level.TimeSeconds - VictimStats.SpawnTime;
			if (TimeAwake<SpawnKillTime)
			{
				Killer.ReceiveLocalizedMessage( class'SmartUTCoolMsg', 5, KillerPRI, VictimPRI );
						
				KillerStats.SpawnKillSpree++;
				if (bShowSpawnKillerGlobalMsg && KillerStats.SpawnKillSpree>2)
					BroadcastLocalizedMessage( class'SmartUTMessage', 10, KillerPRI, VictimPRI );
				
				KillerStats.Score -= SpawnKillPenalty;
				if (bNewSpawnKillScoring && KillerStats.SpawnKillSpree>2)
					KillerStats.Score -= (KillerStats.SpawnKillSpree-2);
	
				// Log
				if (Level.Game.LocalLog!=None)
					Level.Game.LocalLog.LogSpecialEvent( "spawnkill", KillerPRI.PlayerID, VictimPRI.PlayerID, SpawnKillPenalty);				
			}
			else
			{
				KillerStats.SpawnKillSpree = 0; // No spree
			}
		}		
	}
	
	return bPrevent;
}

/*
function bool MutatorBroadcastLocalizedMessage( Actor Sender, Pawn Receiver, out class<LocalMessage> Message, out optional int Switch, out optional PlayerReplicationInfo RelatedPRI_1, out optional PlayerReplicationInfo RelatedPRI_2, out optional Object OptionalObject )
{
	return super.MutatorBroadcastLocalizedMessage( Sender, Receiver, Message, Switch, RelatedPRI_1, RelatedPRI_2, OptionalObject );
}
*/

function bool HandlePickupQuery( Pawn Other, Inventory Item, out byte bAllowPickup )
{
	local SmartUTPRI OtherStats;
	
	if (!bWarmup && Other!=None)
	{
		OtherStats = SmartUTGame.GetStats( Other );
		if (OtherStats!=None)
		{
			// Register pickups
			if (Item.IsA('UT_ShieldBelt'))
				OtherStats.ShieldBelts++;
			else if (Item.IsA('UDamage'))
				OtherStats.Amplifiers++;
			
			// Spawnkill expiration
			if (bSpawnkillDetection && OtherStats.SpawnTime!=0)
			{
				if( Item.IsA( 'TournamentWeapon') || Item.IsA( 'UT_ShieldBelt' ) || Item.IsA( 'UDamage' ) || Item.IsA( 'HealthPack' ) || Item.IsA( 'UT_Invisibility' ) )
			      OtherStats.SpawnTime = 0;
			}

		}
	}
	
	return super.HandlePickupQuery(Other, Item, bAllowPickup);
}

function bool HandleEndGame()
{
	local bool bTied;
	
	if (TeamGamePlus(Level.Game).Teams[0].Score==TeamGamePlus(Level.Game).Teams[1].Score)
		bTied = True;

	if (bForcedEndGame || (bEnableOvertimeControl && !bOvertime && DeathMatchPlus(Level.Game).bTournament))
	{
		bForcedEndGame = False;
		if (bTied)
		{		
			SetEndCamsTiedGame();	
			return True;
		}
	}
	
	if(!bTied)
		CalcSmartUTEndStats();
		
	if (NextMutator!=None)
		return NextMutator.HandleEndGame();
	return False;
}

function ClearStats()
{
	SmartUTGame.ClearStats();
}

function Mutate( string MutateString, PlayerPawn Sender )
{
	local string SoundsString, CMsgsString, MsgsString;
	local SmartUTPRI SenderStats;
	local string Command;

	if ( Left(MutateString, 7)~="SmartUT" || Left(MutateString, 8)~="SmartCTF")
	{
		if (Left(MutateString, 8)~="SmartCTF")
			Command =  Mid(MutateString, 9);
		else
			Command =  Mid(MutateString, 8);
			
		if (Command~="ShowStats" || Command~="Stats")
		{
			SenderStats = SmartUTGame.GetStats( Sender );
			if (SenderStats!=None)
				SenderStats.ToggleStats();
		}
		else if (Command~="ForceStats")
		{
			SenderStats = SmartUTGame.GetStats( Sender );
			if (SenderStats != None)
				SenderStats.ShowStats();
		}
		else if (Command~="ClearStats")
		{
			if (!Sender.PlayerReplicationInfo.bAdmin && Level.NetMode!=NM_StandAlone)
			{
				Sender.ClientMessage( "You need to be logged in as admin to be able to clear the stats." );
			}
			else
			{
				ClearStats();
				Sender.ClientMessage( "Stats cleared." );
			}
		}
		else if (Command~="Rules" || Command~="Points" || Command~="Score" || Command~="Bonus")
		{
			Sender.ClientMessage( "SmartUT Score Settings for:");
			SendSmartUTRules(Sender);
			Sender.ClientMessage( "- Additional features: See Readme!" );
		}
		else if ( Command~="ForceEnd")
		{
			if (!Sender.PlayerReplicationInfo.bAdmin && Level.NetMode!=NM_StandAlone)
			{
				Sender.ClientMessage( "You need to be logged in as admin to force the game to end." );
			}
			else
			{
				BroadcastMessage( Sender.PlayerReplicationInfo.PlayerName @ "forced the game to end." );
				bForcedEndGame = True;
				DeathMatchPlus(Level.Game).EndGame("forced");
			}
		}
		else
		{
			Sender.ClientMessage( "SmartUT" );
			Sender.ClientMessage( "- To toggle stats, bind a key or type in console: 'Mutate SmartUT Stats'" );
			Sender.ClientMessage( "- Type 'Mutate GameInfo' for SmartUT settings." );
			Sender.ClientMessage( "- Type 'Mutate SmartUT Rules' for point system definition." );
			Sender.ClientMessage( "- Type 'Mutate SmartUT ForceEnd' to end a game." );
			if (bEnableOvertimeControl)
				Sender.ClientMessage( "- Type 'Mutate OverTime <On|Off>' for Overtime Control." );
		}
	}
	
	if (Left(MutateString, 8)~="OverTime")
	{
		if( !DeathMatchPlus( Level.Game ).bTournament )
		{
			Sender.ClientMessage( "Not in Tournament Mode: Default Sudden Death Overtime behaviour." );
		}
		else if( !bEnableOvertimeControl )
		{
			Sender.ClientMessage( "Overtime Control is not enabled: Default UT Sudden Death functionality." );
			Sender.ClientMessage( "Admins can use: admin set SmartTDM bEnableOvertimeControl True" );
		}
		else
		{
			if (Left(MutateString, 11)~="OverTime On")
			{
				if (!Sender.PlayerReplicationInfo.bAdmin && Level.NetMode!=NM_StandAlone)
				{
					Sender.ClientMessage( "You need to be logged in as admin to change this setting." );
				}
				else
				{
					bOvertime = True;
					SaveConfig();
					BroadcastLocalizedMessage( class'SmartUTCoolMsg', 3 );
				}
			}
			else if( Left( MutateString, 12 ) ~= "OverTime Off" )
			{
				if( !Sender.PlayerReplicationInfo.bAdmin && Level.NetMode != NM_StandAlone )
				{
					Sender.ClientMessage( "You need to be logged in as admin to change this setting." );
				}
				else
				{
					bOvertime = False;
					SaveConfig();
					BroadcastLocalizedMessage( class'SmartUTCoolMsg', 4 );
				}
			}
			else
			{
				if (Sender.PlayerReplicationInfo.bAdmin || Level.NetMode==NM_StandAlone)
					Sender.ClientMessage( "Usage: Mutate OverTime On|Off" );
				if (!bOvertime)
					Sender.ClientMessage( "Sudden Death Overtime is DISABLED." );
				else
					Sender.ClientMessage( "Sudden Death Overtime is ENABLED (default)." );
				Sender.ClientMessage( "Remember 'Disabled' Setting:" @ bRememberOvertimeSetting );
			}
		}
	}
	else if (Left(MutateString, 8)~="GameInfo" || Left(MutateString, 7)~="CTFInfo")
	{
		// Sounds
		if( bPlayLeadSound ) SoundsString = SoundsString @ "Lead";
		if( bPlay30SecSound ) SoundsString = SoundsString @ "30SecLeft";
		
		// Messages
		
		// Console messages
		if( bShowLongRangeMsg ) CMsgsString = CMsgsString @ "LongRangeKill";
		
		GetSmartUTInfo(SoundsString, MsgsString, CMsgsString);
		if( SoundsString == "" ) SoundsString = "All off";
		if( Left( SoundsString, 1 ) == " " ) SoundsString = Mid( SoundsString, 1 );
		if( MsgsString == "" ) MsgsString = "All off";
		if( Left( MsgsString, 1 ) == " " ) MsgsString = Mid( MsgsString, 1 );
		if( CMsgsString == "" ) CMsgsString = "All off";
		if( Left( CMsgsString, 1 ) == " " ) CMsgsString = Mid( CMsgsString, 1 );
		Sender.ClientMessage( "- Sounds:" @ SoundsString );
		Sender.ClientMessage( "- Msgs:" @ MsgsString );
		Sender.ClientMessage( "- Private Msgs:" @ CMsgsString );
		
		SendSmartUTInfo(Sender);
		
		Sender.ClientMessage( "- bEnhancedMultiKill:" @ bEnhancedMultiKill $ ", Broadcast Level:" @ EnhancedMultiKillBroadcast );
		Sender.ClientMessage( "- bShowExtraInfo:" @ bShowExtraInfo );
		if( bSpawnKillDetection ) Sender.ClientMessage( "- bSpawnKillDetection: True, Global Msg:" @ bShowSpawnKillerGlobalMsg $ ", Penalty:" @ SpawnKillPenalty @ "pts, New SpawnKillScoring:"@bNewSpawnKillScoring );
		else Sender.ClientMessage( "- bSpawnKillDetection: False" );
		Sender.ClientMessage( "- Overtime Control:" @ bEnableOvertimeControl @ "( Type 'Mutate OverTime' )" );
		Sender.ClientMessage( "- Scores: ( Type 'Mutate SmartUT Rules' )");
	}

	super.Mutate( MutateString, Sender );
}

// Override
function SendSmartUTRules(PlayerPawn Sender) {}
function SendSmartUTInfo(PlayerPawn Sender) {}
function GetSmartUTInfo(out string SoundsString, out string MsgsString, out string CMsgsString) {}

function CalcSmartUTEndStats() {}

function PlayLeadSound()
{
	local byte Team;
	local TeamInfo BestTeam;
	local Pawn pn;

	if (bPlayLeadSound)
	{
		if (TeamGamePlus(Level.Game)!=None)
		{
			// Get current best team
			for (Team=0; Team<TeamGamePlus(Level.Game).MaxTeams; Team++)
				if (BestTeam==None || TeamGamePlus(Level.Game).Teams[Team].Score>BestTeam.Score)
					BestTeam = TeamGamePlus(Level.Game).Teams[Team];

			// Not playing or nothing happened
			if (BestTeam.Score==0)
				return;
				
			// Other team with same score as last announced best = losing lead
			if (LastLeadTeam!=None && LastLeadTeam.TeamIndex!=BestTeam.TeamIndex && BestTeam.Score>=LastLeadTeam.Score)
			{
				for (pn=Level.PawnList; pn!=None; pn=pn.NextPawn)
					if (PlayerPawn(pn)!=None && pn.bIsPlayer && (pn.PlayerReplicationInfo.Team==LastLeadTeam.TeamIndex))
						PlayerPawn(pn).ReceiveLocalizedMessage( class'SmartUTAudioMsg', 4);
				LastLeadTeam = None;
			}

			if (LastLeadTeam==None || (BestTeam.Score>LastLeadTeam.Score && BestTeam.TeamIndex!=LastLeadTeam.TeamIndex))
			{
				for (pn=Level.PawnList; pn!=None; pn=pn.NextPawn)
					if (PlayerPawn(pn)!=None && pn.bIsPlayer && (pn.PlayerReplicationInfo.Team==BestTeam.TeamIndex))
						PlayerPawn(pn).ReceiveLocalizedMessage( class'SmartUTAudioMsg', 3);
				LastLeadTeam = BestTeam;
			}

			
		}
		else
		{
		// TODO
		}
	}
}

function ProcessMultiStats(SmartUTPRI OtherStats)
{
	local int Bonus;
	local Pawn Other;
	
	if (OtherStats==None) return;

	if (MultiKillLevelBonus!=0 && OtherStats.MultiLevel>0)
	{
		Bonus = (MultiKillLevelBonus*OtherStats.MultiLevel);	
		OtherStats.Score += Bonus;		
		Other = Pawn(OtherStats.Owner.Owner);	
		if (bShowBonusConsoleMsg && Other!=None)
			Other.ClientMessage("Bonus of"@Bonus@ "point(s) for a"@class'SmartUTEnhancedMultiKillMessage'.static.GetString(OtherStats.MultiLevel)) ;
	}
		
	OtherStats.MultiLevel = 0;
}
	
function ProcessSpreeStats(SmartUTPRI OtherStats)
{
	local int Bonus;
	local string BonusName;
	local int SpreeLevel;
	local Pawn Other;
	
	if (OtherStats==None) return;	
	
	if (SpreeLevelBonus!=0 && OtherStats.FragSpree>=5)
	{
		if (OtherStats.FragSpree<10)
		{
			SpreeLevel = 1;
			BonusName = "Killing Spree";
			//stats.Sprees++;
		}
		else if (OtherStats.FragSpree<15)
		{
			SpreeLevel = 2;
			BonusName = "Rampage";
			//stats.Rampages++;
		}
		else if (OtherStats.FragSpree<20)
		{
			SpreeLevel = 3;
			BonusName = "Dominating";
			//stats.Dominatings++;
		}
		else if (OtherStats.FragSpree<25)
		{
			SpreeLevel = 4;
			BonusName = "Unstoppable";
			//stats.Unstoppables++;
		}
		else if (!bAfterGodLikeMsg && OtherStats.FragSpree>=25)
		{
			SpreeLevel = 5;
			BonusName = "Godlike";
			//stats.Godlikes++;
		}
		else if (bAfterGodLikeMsg && OtherStats.FragSpree<30)
		{
			SpreeLevel = 6;
			BonusName = "'Too easy' Spree";
			//stats.Godlikes++;
		}
		else if (bAfterGodLikeMsg && OtherStats.FragSpree>=35)
		{
			SpreeLevel = 7;
			BonusName = "'Brutalizing' Spree";
			//stats.Godlikes++;
		}	
			
		Bonus = (SpreeLevelBonus*SpreeLevel);
		OtherStats.Score += Bonus;
		Other = Pawn(OtherStats.Owner.Owner);
		if (bShowBonusConsoleMsg && Bonus!=0 && Other!=None )
			Other.ClientMessage("Bonus of"@Bonus@"point(s) for a"@BonusName$"!") ;
	}

	OtherStats.FragSpree = 0;
}

/*
 * Convert a float to a readable string.
 */
function string DitchZeros( float nr )
{
  local string str;

  str = string( nr );
  while( Right( str, 1 ) == "0" )
  {
    str = Left( str , Len( str ) - 1 );
  }
  if( Right( str, 1 ) == "." ) str = Left( str , Len( str ) - 1 );

  return str;
}

//----------------------------------------------------------------------------------------------------------------
//------------------------------------------------ CLIENT FUNCTIONS ----------------------------------------------
//----------------------------------------------------------------------------------------------------------------

/*
 * Render the HUD that is startup logo and FC location.
 * ONLY gets executed on clients.
 */
simulated event PostRender( Canvas C )
{

	if (NextHUDMutator!=None)
		NextHUDMutator.PostRender( C );

  // Get stuff relating to PlayerOwner, if not gotten. Also spawn Font info.
  if( PlayerOwner == None )
  {
    PlayerOwner = C.Viewport.Actor;
    MyHUD = ChallengeHUD( PlayerOwner.MyHUD );

    pTGRI = TournamentGameReplicationInfo( PlayerOwner.GameReplicationInfo );
    pPRI = PlayerOwner.PlayerReplicationInfo;
    MyFonts = MyHUD.MyFonts;
  }
  
}


/*
 * Executed on the client when that player joins the server.
 */
simulated function ClientJoinServer( Pawn Other )
{
	if( PlayerPawn( Other ) == None || !Other.bIsPlayer )
		return;
	
	Other.ClientMessage( "Running SmartUT " $ Version $ ". Type 'Mutate SmartUT' in the console for info." );
	
	// Since this gets called in the HUD it needs to be changed clientside.
	if (SmartUTGame.bPlay30SecSound )
		class'TimeMessage'.default.TimeSound[5] = sound'Announcer.CD30Sec';
}

/*
 * Clientside settings that need to be set for the first time, checking for welcome message and
 * end of game screen.
 */
simulated function Tick( float delta )
{
	local Pawn p;

	if( Level.NetMode == NM_DedicatedServer || Role == ROLE_Authority )
	{
		if (bCheckReady)
		{
			if (DeathMatchPlus(Level.Game).bTournament && DeathMatchPlus(Level.Game).CountDown>0)
			{
				for ( P=Level.PawnList; P!=None; P=P.NextPawn )
					if (P.IsA('PlayerPawn') && !P.IsA('Spectator') && PlayerPawn(P).bReadyToPlay && P.PlayerReplicationInfo.HasFlag == None)
						P.PlayerReplicationInfo.HasFlag = ReadyFlag;
			}
			else
			{
				bCheckReady = false;
				for ( P=Level.PawnList; P!=None; P=P.NextPawn )
					P.PlayerReplicationInfo.HasFlag = None;					
				if (ReadyFlag != None)
					ReadyFlag.Destroy();
				ReadyFlag = None;
			}
		}
	}

  // Execute on client
  if( Level.NetMode != NM_DedicatedServer )
  {
    if( SmartUTGame == None )
    {
      ForEach AllActors(class'SmartUTGameReplicationInfo', SmartUTGame) break;
      if( SmartUTGame == None ) return;

      if( !SmartUTGame.bServerInfoSetServerSide && SmartUTGame.DefaultHUDType != None ) // client side required
      {
        class<ChallengeHUD>( SmartUTGame.DefaultHUDType ).default.ServerInfoClass = SmartUTServerInfoClass;
        Log( "Notified HUD (clientside," @ SmartUTGame.DefaultHUDType.Name $ ") to use SmartUT ServerInfo.", 'SmartUT' );
      }
    }

    if( !SmartUTGame.bInitialized ) return;

    if( !bHUDMutator ) RegisterHUDMutator();

    if( PlayerOwner != None )
    {
      if( !bClientJoinPlayer )
      {
        bClientJoinPlayer = True;
        ClientJoinServer( PlayerOwner );
      }

      // If Game is over, bring up F3.
      if( PlayerOwner.GameReplicationInfo.GameEndedComments != "" && !bGameEnded )
      {
        bGameEnded = True;
        PlayerOwner.ConsoleCommand( "mutate SmartUT ForceStats" );
      }
    }
  }
 
}

function SetEndCamsTiedGame()
{
	local Pawn pn, Best;
	local PlayerPawn Player;

	// Find Individual Winner
	for( pn = Level.PawnList ; pn != None ; pn = pn.NextPawn )
	{
		if (pn.bIsPlayer && ((Best==None)||(pn.PlayerReplicationInfo.Score>Best.PlayerReplicationInfo.Score)))
			Best = pn;
	}

	DeathMatchPlus(Level.Game).GameReplicationInfo.GameEndedComments = GameTieMessage;
	DeathMatchPlus(Level.Game).EndTime = Level.TimeSeconds + 3.0;

  for( pn = Level.PawnList ; pn != None ; pn = pn.NextPawn )
  {
    Player = PlayerPawn( pn );
    if( Player != None )
    {
      Player.bBehindView = True;
      if( Player == Best ) Player.ViewTarget = None;
      else Player.ViewTarget = Best;

      Player.ClientPlaySound( sound'CaptureSound', , true );
      Player.ClientGameEnded();
    }
    pn.GotoState( 'GameEnded' );
  }

  DeathMatchPlus(Level.Game).CalcEndStats();
  CalcSmartUTEndStats();
}

/*
 * For showing the Logo a Timer is used instead of Ticks so its equal for each tickrate.
 * On the server it keeps track of some replicated data and whether a Tournament game is starting.
 */
simulated function Timer()
{
	local bool bReady;
	local Pawn pn;
	local SmartUTPRI OtherStats;

	super.Timer();

	// Client
	if (Level.NetMode!=NM_DedicatedServer)
	{
		if (Role!=ROLE_Authority)
			SetTimer(0.0, False); // client timer off
        else
			SetTimer(1.0, True); // standalone game? keep timer running for bit below.
	}
	
	// Server - 1 second timer. infinite.
	if( Level.NetMode == NM_DedicatedServer || Role == ROLE_Authority )
	{
		SmartUTGame.TickRate = int( ConsoleCommand( "get IpDrv.TcpNetDriver NetServerMaxTickRate" ) );

	    if (!bTournamentGameStarted && DeathMatchPlus(Level.Game).bTournament)
	    {
	      if( DeathMatchPlus( Level.Game ).bRequireReady && DeathMatchPlus( Level.Game ).CountDown > 0
	       && ( DeathMatchPlus( Level.Game ).NumPlayers == DeathMatchPlus( Level.Game ).MaxPlayers || Level.NetMode == NM_Standalone )
	       && DeathMatchPlus( Level.Game ).RemainingBots <= 0 )
	      {
	        bReady = True;
	        for( pn = Level.PawnList; pn != None; pn = pn.NextPawn )
	        {
	          if( pn.IsA( 'PlayerPawn' ) && !pn.IsA( 'Spectator' ) && !PlayerPawn( pn ).bReadyToPlay )
	          {
	            bReady = False;
	            break;
	          }
	        }
	      }

			if( bReady )
			{
		        bTournamentGameStarted = True;
		        TournamentGameStarted();
			}
		}

		// Register multikills
		if (MultiKillLevelBonus!=0)
		{
			foreach AllActors(class'SmartUTPRI', OtherStats)
				if (Level.TimeSeconds-OtherStats.LastKillTime>=3)
					ProcessMultiStats(OtherStats);
		}
	
	}
}

defaultproperties
{
    Version="01"
    GameTieMessage="The game ended in a tie!"
    SmartUTEndStatsClass=Class'SmartUTEndStats'
    SmartUTGameReplicationInfoClass=Class'SmartUTGameReplicationInfo'
    SmartUTScoreBoardClass=Class'SmartUTScoreBoard'
    SmartUTServerInfoClass=Class'SmartUTServerInfo'
    RedTeamColor=(R=255,G=0,B=0,A=0),
    BlueTeamColor=(R=0,G=128,B=255,A=0),
    White=(R=255,G=255,B=255,A=0),
    Gray=(R=128,G=128,B=128,A=0),
    bEnabled=True
    bAfterGodLikeMsg=True
    EnhancedMultiKillBroadcast=3
    bServerInfo=True
    bShowExtraInfo=True
    bSpawnKillDetection=True
    bShowSpawnKillerGlobalMsg=True
    SpawnKillTime=3.50
    bStatsDrawFaces=True
    SuicidePenalty=1
    SpawnKillPenalty=1
    bPlayLeadSound=True
    bPlay30SecSound=True
    bOverTime=True
    bAlwaysRelevant=True
    RemoteRole=2
}
