
package MasterServer::Util;

use strict;
use warnings;
use Data::Dumper 'Dumper';
use POSIX qw/strftime/;
use Exporter 'import';

our @EXPORT = qw| log secure validate valid_address hasValidKey |;


################################################################################
# Log to file and print to screen.
# args: $self, $out (-1 none, 0 log only, 1 stdout, $msg
################################################################################
sub log {
  my ($self, $msg) = @_;
  
  my $time = strftime('%Y-%m-%d %H:%M:%S',localtime);
  
  # FIXME
  print "$msg\n";
  
  # temporarily disable the warnings-to-log, to avoid infinite recursion if
  # this function throws a warning.
  my $old = $SIG{__WARN__};
  $SIG{__WARN__} = undef;

  chomp $msg;
  $msg =~ s/\n/\n  | /g;
  if($self->{logfile} && open my $F, '>>:utf8', $self->{logfile}) {
    flock $F, 2;
    seek $F, 0, 2;
    print $F "[$time]\t$msg\n";
    flock $F, 4;
    close $F;
  }
  $SIG{__WARN__} = $old;
}

################################################################################
# generate a random string of 6 characters long for the \secure\ challenge
# args: none
################################################################################
sub secure {
  # spit out a random string, only uppercase characters
  #my @c = ('a'..'z', 'A'..'Z', 0..9);
  my @c = ('A'..'Z');
  my $s = "";
  $s .= $c[rand @c] for 1..6;
  
  # return random string
  return $s;
}

################################################################################
# calculate the \validate\ response for the \secure\ challenge. 
# args: gamename, \secure\ string
# !! requires cipher hash to be configured in config!
################################################################################
sub validate {
  my ($self, $gn, $s)  = @_;
  
  # get cipher from gamename
  my $cip = $self->{cipher}{$gn} || "000000";
  
  if (length $s > 16) {
    return "0"}
  
  # return the expected validate response
  return `$self->{gsmsalg_path} $s $cip`;
}

# returns 1 on valid key, 0 on invalid
sub hasValidKey {
  my ($self, $gamename, $secure, $validate) = @_;
  
  # more than 8 characters is weird/not standard
  if (length $secure > 8) {
    return 0}

  # ignore games that do difficult or stupid with game keys
  if ((join " ", @{$self->{ignore_key}}) =~ m/$gamename/i){
    $self->log("[IGN] > Ignored validation for $gamename.");
    return 1;
  }

  # get cipher from gamename
  my $cipher = $self->{cipher}{$gamename} || "000000";
  
  # return the expected validate response
  return (`$self->{gsmsalg_path} $secure $cipher` eq $validate) || 0;
}

################################################################################
##
##   Valid Address
##
##   Verify whether a given domain name or IP address and port are valid.
##   returns the valid ip-address + port, or 0 when not.
################################################################################
sub valid_address {
  my ($self, $h) = @_;

  # split up address and port
  my ($a, $p) = ($h =~ m/:/) ? $h =~ /(.*):(.*)/ : ($h,0);
  
  # resolve hostname when needed
  if($a =~ /[a-zA-Z]/g) {
     my $raw_addr = (gethostbyname($a))[4];
     my @octets = unpack("C4", $raw_addr);
     $a = join(".", @octets);
  }
  
  # check if IP and port are in valid range
  $a = ($a =~ '\b(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b') ? $a : 0;
  $p = (0 < $p && $p <= 65535) ? $p : 0;
  
  # exclude addresses where we don't want people sniffing
  for (qw|192.168.(.\d*).(.\d*) 127.0.(.\d*).(.\d*) 10.0.(.\d*).(.\d*)|){$a = 0 if ($a =~ m/$_/)}

  return ($a, $p);
}

1;
