/***************************************************************************************************
 *
 *  SendTo standalone by {LSN}Meindratheal
 *
 *  $CLASS        STWMut
 *  $VERSION      1.05 (28-04-2011 18:07)
 *  $AUTHOR       {LSN}Meindratheal
 *  $DESCRIPTION  SendTo Mutator. Configuration resides in the server's INI file.
 *
 **************************************************************************************************/
class STWMut extends Mutator config(system);

//Global
var config bool bShowCommandsOnJoin;
var config bool bShowHelpMessage;

var config byte bAdvertise[64]; //Show to anyone
var config byte bHideCmd[64]; //Hide the say/teamsay when used
var config string Message[64]; //Help message
var config string LongCmd[64]; //Long command string
var config string ShortCmd[64]; //Short command string
var config string TargetURL[64]; //URL to link to.

//Internal
var int NumSites; //How many websites are in the list.
var int DeathsForWelcome; //How many deaths should a player have for the message to show? For MH.

var STWClient ClientList; //First remote client instance.

var string Version;

//Secondary array for validation of configuration
var byte bA[64];
var byte bH[64];
var string M[64];
var string LC[64];
var string SC[64];
var string TU[64];

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called as soon as this actor is created.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function PreBeginPlay()
{
	ValidateConfig();
	Level.Game.RegisterMessageMutator(self);
	//Find the DeathsForWelcome.
	if(instr(caps(string(level.game)), "MONSTERHUNT") >= 0)
	{
		//Monsterhunt uses deaths as lives, so broadcasting at
		//0 lives will only work on game over :D
		if(bool(Level.Game.GetPropertyText("bUseLives")))
		{
			DeathsForWelcome = int(Level.Game.GetPropertyText("Lives"));
		}
		else
		{
			DeathsForWelcome = 0;
		}
	}
	else
	{
		DeathsForWelcome = 0;
	}
	Log("************************");
	Log("*     SendTo v"$Version$"      *");
	Log("*      is active!      *");
	Log("*  "$NumSites$" links detected!   *");
	Log("************************");
	SaveConfig();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Handles a potential command message.
 *  $PARAM        sender     PlayerPawn that has send the message in question.
 *  $PARAM        msg        Message send by the player, which could be a command.
 *  $REQUIRE      sender != none
 *
 **************************************************************************************************/
function HandleMsgCommand(PlayerPawn Sender, string Msg)
{
	local int i;
	Msg = class'STWUtils'.static.trim(Msg);

	if(Sender == None || Msg == "")
	{
		//Not enough information.
		return;
	}

	//Built-in commands. Cannot be overridden.
	if(Msg ~= "!STWHelp")
	{
		ShowConsoleHelp(Sender);
		return;
	}
	if(Msg ~= "!List" || Msg ~= "!Links" || Msg ~= "!ListLinks")
	{
		ShowLinkList(Sender);
		return;
	}

	//We can't do a switch, that only works with constants.
	for(i = 0; i < NumSites; i++)
	{
		if(Msg ~= LongCmd[i] || Msg ~= ShortCmd[i])
		{
			//It's this one.
			WebConnect(Sender, TargetURL[i]);
			return;
		}
	}
	//Not a valid STW command.
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player has sent a mutate call to the server.
 *  $PARAM        mutateString  Mutator specific string (indicates the action to perform).
 *  $PARAM        sender        Player that has send the message.
 *
 **************************************************************************************************/
function Mutate(string MutateString, PlayerPawn Sender)
{
	local string Msg;

	Msg = MutateString;

	if(left(Msg, 4) ~= "STW ")
	{
		Msg = Mid(Msg, 4);
		if(left(Msg, 1) != "!")
		{
			Msg = "!"$Msg; //Add an exclamation mark if not used in MS.
		}
		HandleMsgCommand(Sender, Msg);
	}
	else if(Msg ~= "STWHelp" || Msg ~= "!STWHelp")
	{
		HandleMsgCommand(Sender, "!STWHelp");
	}
	else if(Msg ~= "List" ||Msg ~= "!List" || Msg ~= "Links" || Msg ~= "!Links" ||
		Msg ~= "ListLinks" || Msg ~= "!ListLinks")
	{
		HandleMsgCommand(Sender, "!List");
	}

	Super.Mutate(MutateString, Sender);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is sent to player.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        pri       Player replication info of the sending player.
 *  $PARAM        s         The message that is to be send.
 *  $PARAM        type      Type of the message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorTeamMessage(Actor sender, Pawn receiver, PlayerReplicationInfo pri,
                                 coerce string s, name type, optional bool bBeep)
{
	local bool bHideMsg;

	// Check for commands.
	if (sender != none && sender.isA('PlayerPawn') && sender == receiver &&
	    (type == 'Say' || type == 'TeamSay'))
	{
		HandleMsgCommand(PlayerPawn(sender), s);
	}

	//Moved here because other clients call this as well, hence the reason they weren't getting blocked.
	bHideMsg = HideCmd(s);

	// Allow other message mutators to do their job.
	if (nextMessageMutator != none)
	{
		return	(!bHideMsg &&
			nextMessageMutator.mutatorTeamMessage(sender, receiver, pri, s, type, bBeep));
	}
	return !bHideMsg;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Hooked into the message mutator chain so commands can be detected. This function
 *                is called if a message is send to player. Spectators that use say (not teamsay)
 *                seem to be calling this function instead of mutatorTeamMessage.
 *  $PARAM        sender    The actor that has send the message.
 *  $PARAM        receiver  Pawn receiving the message.
 *  $PARAM        msg       The message that is to be send.
 *  $PARAM        bBeep     Whether or not to make a beep sound once received.
 *  $PARAM        type      Type of the message that is to be send.
 *  $RETURN       True if the message should be send, false if it should be suppressed.
 *  $OVERRIDE
 *
 **************************************************************************************************/
function bool mutatorBroadcastMessage(Actor sender, Pawn receiver, out coerce string msg,
                                      optional bool bBeep, out optional name type)
{
	local bool bIsSpecMessage, bHideMsg;
	local PlayerReplicationInfo senderPRI;

	// Get sender player replication info.
	if (sender != none && sender.isA('Pawn')) {
		senderPRI = Pawn(sender).playerReplicationInfo;
	}

	// Check if we're dealing with a spectator chat message.
	bIsSpecMessage = senderPRI != none && sender.isA('Spectator') &&
	                 left(msg, len(senderPRI.playerName) + 1) ~= (senderPRI.playerName $ ":");

	// Check for commands.
	if (bIsSpecMessage && sender == receiver)
	{
		HandleMsgCommand(PlayerPawn(sender), mid(msg, len(senderPRI.playerName) + 1));

	}
	//Moved here because other clients call this as well, hence the reason they weren't getting blocked.
	if(senderPRI != None)
	{
		bHideMsg = HideCmd(mid(msg, len(senderPRI.playerName) + 1));
	}

	// Allow other message mutators to do their job.
	if (nextMessageMutator != none)
	{
		return	(!bHideMsg &&
			nextMessageMutator.mutatorBroadcastMessage(sender, receiver, msg, bBeep, type));
	}
	return !bHideMsg;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Called when a player (re)spawns and allows us to modify the player.
 *  $PARAM        other  The pawn/player that has (re)spawned.
 *  $REQUIRE      other != none
 *  $WARNING      May not be compatible with MonsterHunt.
 *
 **************************************************************************************************/
function ModifyPlayer(Pawn Other)
{
	if(Other.IsA('PlayerPawn') && Other.PlayerReplicationInfo.Deaths == DeathsForWelcome)
	{
		ShowWelcomeMsg(PlayerPawn(Other));
	}
	Super.ModifyPlayer(Other);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the welcome message to a player. May also broadcast the help.
 *  $PARAM        P   The PlayerPawn requesting the help.
 *  $REQUIRE      P != None
 *
 **************************************************************************************************/
function ShowWelcomeMsg(PlayerPawn P)
{
	local int i;
	if(bShowHelpMessage)
	{
		P.ClientMessage("This server runs STW v"$Version$",");
		P.ClientMessage("say !STWHelp for a list of available commands.");
	}
	if(bShowCommandsOnJoin)
	{
		//for(i = 0; i < ArrayCount(TargetURL); i++)
		for(i = 0; i < NumSites; i++)
		{
			if(bAdvertise[i] > 0 && Message[i] != "")
			{
				//Show this one.
				P.ClientMessage(Message[i]);
			}
		}
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the help commands to a player.
 *  $PARAM        P   The PlayerPawn requesting the help.
 *  $REQUIRE      P != None
 *
 **************************************************************************************************/
function ShowConsoleHelp(PlayerPawn P)
{
	local int i;
//	for(i = 0; i < ArrayCount(TargetURL); i++)
	for(i = 0; i < NumSites; i++)
	{
		if(bAdvertise[i] > 0 && Message[i] != "")
		{
			//Show this one.
			P.ClientMessage(Message[i]);
		}
	}
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Shows the link list to a player.
 *  $PARAM        P   The PlayerPawn requesting the help.
 *  $REQUIRE      P != None
 *
 **************************************************************************************************/
function ShowLinkList(PlayerPawn P)
{
	local int i;
	local bool bAdmin;

	bAdmin = P.bAdmin || P.PlayerReplicationInfo.bAdmin;

	P.ClientMessage("Link list:");
	for(i = 0; i < NumSites; i++)
	{
		if(bAdvertise[i] > 0 || bAdmin)
		{
			P.ClientMessage(LongCmd[i] @ " " @ ShortCmd[i] @ " " @ TargetURL[i]);
		}
	}
}



/***************************************************************************************************
 *
 *  $DESCRIPTION  Checks whether a message should be hidden
 *  $RETURN       True if the message should be hidden, false if it should be sent.
 *
 **************************************************************************************************/
function bool HideCmd(string Cmd)
{
	local int idx;
	idx = GetCmdIndex(Cmd);

	return (idx >= 0 && idx < NumSites && bHideCmd[idx] > 0);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Finds the index of a command.
 *  $PARAM        Cmd    The command whose index we need.
 *  $RETURN       The index of the command, or -1 if not found.
 *
 **************************************************************************************************/
function int GetCmdIndex(string Cmd)
{
	local int i;
	local bool bFound;
//	while(i < ArrayCount(TargetURL))
	while(i < NumSites)
	{
		//Log("GCI: Comparing" @ Cmd @ "with" @ LongCmd[i] @ "and" @ ShortCmd[i]);
		if(LongCmd[i] ~= Cmd || ShortCmd[i] ~= Cmd)
		{
			return i;
		}
		i++;
	}
	//Not found.
	return -1;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Sends the command to the client. Not necessarily efficient,
 *                but I suck at replication issues.
 *  $PARAM        Sender     PlayerPawn that has called the command
 *  $PARAM        URL        Target URL
 *  $REQUIRE      sender != none
 *
 **************************************************************************************************/
function WebConnect(PlayerPawn Sender, string URL)
{
	local STWClient C, Target;
	for(C = ClientList; C != None; C = C.NextClient)
	{
		if(C.Owner == Sender)
		{
			Target = C;
			break;
		}
	}
	if(Target == None)
	{
		Target = Spawn(class'STWClient', Sender);
		Target.Link(self);
	}
	Target.ClientWebConnect(URL);
}



/***************************************************************************************************
 *
 *  $DESCRIPTION  Ensures the configuration is valid.
 *
 **************************************************************************************************/
function ValidateConfig()
{
	local int i;

	NumSites = 0;

	for(i = 0; i < ArrayCount(TargetURL); i++)
	{
		//Load data into the next secondary slot. If the last loop found invalid data, will overwrite that.
		bA[NumSites] = bAdvertise[i];
		bH[NumSites] = bHideCmd[i];
		M[NumSites] = class'STWUtils'.static.trim(Message[i]);
		LC[NumSites] = class'STWUtils'.static.trim(LongCmd[i]);
		SC[NumSites] = class'STWUtils'.static.trim(ShortCmd[i]);
		TU[NumSites] = ValidateURL(TargetURL[i]);
		if((bA[NumSites] > 0 && M[NumSites] == "")|| LC[NumSites] == "" || TU[NumSites] == "")
		{
			//Not enough information.
			//continue;
		}
		else
		{
			if(left(LC[NumSites], 1) != "!")
			{
				LC[NumSites] = "!"$LC[NumSites];
			}
			//Check for conflicts with commands. Does not pick up on conflicts with other mods (e.g. Nexgen)!
			if(ConflictingLongCmd(NumSites))
			{
				MLog("[STW] Warning: conflicting command:" @ LC[NumSites]);
			}
			if(SC[NumSites] == "")
			{
				//Take the first letter of the long command.
				SC[NumSites] = Left(LC[NumSites], 2); //Include !
			}
			//Check for conflicts with commands. Does not pick up on conflicts with other mods (e.g. Nexgen)!
			if(ConflictingShortCmd(NumSites))
			{
				MLog("[STW] Warning: conflicting command:" @ SC[NumSites]);
			}
			NumSites++;
		}
	}
	//Validated, now copy back and save config.
	for(i = 0; i < ArrayCount(TargetURL); i++)
	{
		bAdvertise[i] = bA[i];
		bHideCmd[i] = bH[i];
		Message[i] = M[i];
		LongCmd[i] = LC[i];
		ShortCmd[i] = SC[i];
		TargetURL[i] = TU[i];
	}
	SaveConfig();
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Ensures a URL is valid.
 *  $PARAM        U    The URL to check
 *  $RETURN       The clean URL.
 *
 **************************************************************************************************/
function string ValidateURL(string URL)
{
	local int PIdx;
	local string Protocol, Address;

	if(URL == "")
	{
		return "";
	}

	//Split apart the protocol and the website.
	PIdx = instr(URL, "://");
	if(PIdx == -1)
	{
		//No protocol.
		Protocol = "http";
		Address  = URL;
	}
	else if(PIdx == 0)
	{
		//For some reason, in the format ://www.website.com
		Protocol = "http";
		Address = Mid(URL, 3);
	}
	else
	{
		//In the format HTTP://www.website.com
		Protocol = Left(URL, PIdx);
		Address = Mid(URL, PIdx + 3);
	}
	if(Address == "" || Protocol == "")
	{
		return "";
	}

	return class'STWUtils'.static.trim(Protocol $ "://" $ Address);
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check if a long command conflicts with any others.
 *  $PARAM        CmdIdx   The idx of the command we are checking for.
 *  $RETURN       True if there is a conflict, False otherwise.
 *
 **************************************************************************************************/
function bool ConflictingLongCmd(int CmdIdx)
{
	local bool bConflict;
	local int i;
	while(!bConflict && i < CmdIdx)
	{
		//Compare with temp array!
		if(LC[CmdIdx] ~= LC[i])
		{
			bConflict = True;
		}
		i++;
	}
	return bConflict;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Check if a short command conflicts with any others.
 *  $PARAM        CmdIdx   The idx of the command we are checking for.
 *  $RETURN       True if there is a conflict, False otherwise.
 *
 **************************************************************************************************/
function bool ConflictingShortCmd(int CmdIdx)
{
	local bool bConflict;
	local int i;
	while(!bConflict && i < CmdIdx)
	{
		//Compare with temp array!
		if(SC[CmdIdx] ~= SC[i])
		{
			bConflict = True;
		}
		i++;
	}
	return bConflict;
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Logs a message.
 *  $PARAM        M    The string to log.
 *
 **************************************************************************************************/
function MLog(coerce string M)
{
	Log(M, 'STWMut');
}

/***************************************************************************************************
 *
 *  $DESCRIPTION  Default properties block.
 *
 **************************************************************************************************/
defaultproperties
{
  bShowCommandsOnJoin=True
  bShowHelpMessage=True
  bHidden=True
  Version="0.7"
}